/*
 Low-cost UECS nutrient supplier
 LU-NUTSUP By T.Hoshi, Kindai Univ.
 2017.09.15 version 1.0
 2017.10.31 強制給液時間間隔を設定していた時、日付が変わって
 前回給液時刻が0時にリセットされてしまうので、夜明け時に強制
 給液してしまった。それを防止するために、対策を行った。ver. 1.0a
 2018.8.10 太陽電池のシャント抵抗を3.3Ωにしたため、日射換算値を変更した。
 2019.1.16  3点の改良を実施　version 1.1
 　・1回の給液時間を秒単位で設定可能にする。
 　・給液定数の設定単位をL/株/分からcc/株/分にする。初期値を10に。
 　・時刻を外部参照できるようにする。内蔵クロック不要に。
*/


// このプログラムは、無償提供する代わりに、動作結果に関する
// 一切の保証は致しません。自己責任でお使いください。また、
// 改造も自由ですが、公刊等をされる場合は、「近畿大学　星　
// 岳彦が開発した○○〇プログラムを用いた」と引用して下さい
// ますと、今後の研究開発の励みになりますので、どうぞよろ
// しくお願いします。　　　　　2019.3.4 Takehiko Hoshi
// 
// このシールドはArduino MEGA 2560 + Ethernet Shiedld2の
// 組み合わせで動作させることを前提にしています。
// プログラムを書き込むときには、Arduino MEGA 2560だけ
// の状態で書き込んでください。
// Ethernet Sheild 2基板のシールのmacアドレスに変更して
// ください。このプログラムをma[で検索すると付近にあり
// ます。macアドレスとは。90:A2:DA:0F:xx:xx のような8個
// の16進数です。これをばらして0xの後に転記してください。

// 出力の割り付け状態
// Dout0 なし
// Dout1 なし
// Dout2 給液信号　ONの時だけ給液
// Dout3 なし
 
#include <Wire.h>
#include <SPI.h>
#include <Ethernet2.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <EEPROM.h>
// UECS用のライブラリUARDECSのインストール
// https://uecs.org/arduino/uardecs.html が必要です。
#include <Uardecs.h>

//////////////////////
// 定数関連定義
//////////////////////

// 計測信号線の定義
//センサ・SD信号線定義
//analog
const uint8_t srsPin  =  0 ; //太陽電池 光関係
// I2Cデバイスアドレスの定義
#define RTC_ADRS  0x32 //リアルタイムクロック
// Arduino Ethernet Shield2のMacアドレス
const byte ma[6] = { 0x90, 0xA2, 0xDA, 0x10, 0xDF, 0x76 } ;

//
//IPアドレスをリセットするピンの割付
//
//Pin ID. This pin is pull-upped automatically.
const byte U_InitPin = 57 ; // for Arduino MEGA 2560 
//const byte U_InitPin = 17 ; // for Arduino Ethernet
//This shields is assigned as A3 for reset pin.
//When U_InitPin status equals this value,IP address is set "192.168.1.7".
//(This is added Ver3 over)
const byte U_InitPin_Sense= HIGH ;  

/////////////////////////////
// ノードの基礎情報の設定
/////////////////////////////

const char U_name[] PROGMEM =   "LU-NUTSUP11";
const char U_vender[] PROGMEM = "Hoshi-lab.";
const char U_uecsid[] PROGMEM = "011009001002";
const char U_footnote[] PROGMEM = "Product with UARDECS7.5 by <A HREF=\"http://hoshi-lab.info/\">Kindai, hoshi-lab.</A>";
char U_nodename[20] = "LU-NUTSUP11";
UECSOriginalAttribute U_orgAttribute;

////////////////////////////////////////////
// 培養液供給ノードの変数設定とライブラリ
////////////////////////////////////////////

// 入出力ポートの定義
const int port[4] = { 7, 8, 5, 6 } ;
// ウオッチドッグタイマーのリセットポート
const int wdt = 9 ; // 0 = clear timer
// ディジタル出力値の一時記憶装置のトリガポート
const int ratch = 3 ; //  0->1 ratch
// ディジタル入力値のイネーブルポート
const int in_out = 2 ; // in = 1 out =0

// 作業変数
int i ; // ループ用変数
float mv ; // AD変換用作業変数
signed long intsupply ; // 1分間の秒ごとの給液動作時の給液定数積算値
                        // 総給液量の計算に必要

// 自動給液時間の残(秒)
signed long rNutSec ; //毎秒これが0以上なら給液する

//Define variables for Web interface
signed long W_supply ; // 培養液供給
signed long W_mode ; // 給液モード
signed long W_modeset ; // 給液モード設定
signed long W_ntconst ; // 給液定数 ml/株/分 0.1 ml単位(1/10)
signed long W_onesp ; // 給液時間 単位:秒
signed long W_ntspsrint ; // 給液積算日射
signed long W_ntfcspint ; // 強制給液間隔
signed long W_fctime1 ; // 強制給液時刻1
signed long W_fctime2 ; // 強制給液時刻2
signed long W_time_in ; // 内部時刻
signed long W_setdtf ; // 日時設定・時計モード0:外部、1:内部、2:内部設定
signed long W_useoutsrf ; // 外部日射使用
signed long W_sr_in ; // 内部日射量
signed long W_pnrtime ; // 前回給液時刻
signed long W_srint_pr ; // 現在積算日射
signed long W_srint_td ; // 本日積算日射
signed long W_amnt_td ; // 本日給液量
signed long W_amnt_yd ; // 昨日給液量
signed long W_amnt_all ; // 総給液量

// ウオッチドッグタイマーのクリア(2秒以内に1回の実行が必要)
void clearWdt() {
  digitalWrite(wdt, LOW) ;
  delayMicroseconds(100) ;
  digitalWrite(wdt, HIGH) ;
}

// ディジタル値を出力する d[0]-d[3]の値 0はOFF、1はON
void dout(int *d) {
  digitalWrite(in_out, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], OUTPUT) ;
    digitalWrite(port[i], d[i]) ;
  }
  delayMicroseconds(1) ;
  digitalWrite(ratch, HIGH) ;
  delayMicroseconds(1) ; 
  digitalWrite(ratch, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], INPUT) ;
  }
  digitalWrite(in_out, HIGH) ;
}

// ディジタル出力を全てOFFにする
void clearDout() {
  int d[4] = { 
    LOW, LOW, LOW, LOW
  } ;
  dout(d) ;
}

// 養液供給
void nutrientON() {
  int d[4] = { 
    LOW, LOW, HIGH, LOW
  } ;
  dout(d) ;
  W_supply = 1 ;
}
  
// 養液停止
void nutrientOFF() {
  clearDout() ;
  W_supply = 0 ;
}

// ディジタル入力値の読み込み d[0]-d[3]に値を代入
void din(int *d) {
  digitalWrite(in_out, HIGH) ;
  delayMicroseconds(1) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], INPUT) ;
    delayMicroseconds(1) ;
    d[i] = digitalRead(port[i]) ;
  }
}

/////////
// RTC
/////////

//RTC関係グローバル変数
signed long v_date ; //日付
signed long v_time ; //時刻
byte seconds,minutes,pminutes,hours,days,months,years;
// 秒カウンター
int Csec ; // 1分を計時するため

//RTCから日時読み込み
void RTCread() {
  Wire.requestFrom(RTC_ADRS,8);
  Wire.read();
  seconds = Wire.read();
  minutes = Wire.read();
  hours = Wire.read();
  v_time = ((long)(hours >> 4) * 10 + (long)(hours & 0xF)) * 100 + ((long)(minutes >> 4) * 10 + (long)(minutes & 0xF))  ;
  Wire.read();
  days = Wire.read();
  months = Wire.read();
  years = Wire.read();
  v_date = ((long)(years >> 4) * 10 + (long)(years & 0xF)) * 10000 + ((long)(months >> 4) * 10 + (long)(months & 0xF)) * 100 + (long)(days >> 4) * 10 + (long)(days & 0xF) ;
}

//RTC日時書き込み
void RTCset() {
  byte bbuf ;
  Wire.beginTransmission(RTC_ADRS);
  Wire.write(0x00);
  Wire.write(0x00);  //sec
  bbuf = (byte)(v_time % 10) + (byte)(v_time / 10 % 10 * 16) ; //1と10の位
  Wire.write(bbuf);  //min
  bbuf = (byte)(v_time / 100 % 10) + (byte)(v_time / 1000 % 10 * 16) ; //100と1000の位
  Wire.write(bbuf);  //hour
  Wire.write(0x00);  //week 0:日曜 １:月曜 ・・・ 6:土曜
  bbuf = (byte)(v_date % 10) + (byte)(v_date / 10 % 10 * 16) ; //1と10の位
  Wire.write(bbuf);  //day
  bbuf = (byte)(v_date / 100 % 10) + (byte)(v_date / 1000 % 10 * 16) ; //100と1000の位
  Wire.write(bbuf);  //month
  bbuf = (byte)(v_date / 10000 % 10) + (byte)(v_date / 100000 % 10 * 16) ; //10000と100000の位
  Wire.write(bbuf);  //year
  Wire.endTransmission();
  delay(1);
}

//time1 - time2 の経過分数
signed long elapsMin(signed long time1, signed long time2) {
  signed long wk ;
  wk = ((time1 / 100 * 60) + (time1 % 100)) - ((time2 / 100 * 60) + (time2 % 100)) ;
  if(wk < 0) wk += 1440 ;
  return wk ;
}

//////////////////////////////////
// 計測制御基本ライブラリ
//////////////////////////////////

// 太陽電池で日射量測定 
signed long SRmeasure() {
  mv = (float)analogRead(srsPin) *  5000 / 1023 ;
  if(mv < 0) mv = 0.0 ;
   // solar radiation convert
   // for 3.3Ω 2018.8.10
  mv = 0.6704 * mv * 2 ; // W m-2
  return  (signed long)mv ; 
}

//////////////////////////////////
// html page setting
//////////////////////////////////

// Define Strings
const char S_supply[] PROGMEM =  "養液";
const char S_modedp[] PROGMEM =  "動作モード";
const char S_modeset[] PROGMEM =  "動作設定";
const char S_const[] PROGMEM =  "給液定数";
const char S_time[] PROGMEM =  "給液時間";
const char S_spintsr[] PROGMEM =  "給液積算日射";
const char S_intfcntsup[] PROGMEM =  "強制給液";
const char S_fcntsup1[] PROGMEM =  "強制給液時刻1";
const char S_fcntsup2[] PROGMEM =  "強制給液時刻2";
const char S_insidedt[] PROGMEM =  "日付";
const char S_insidetm[] PROGMEM =  "時刻";
const char S_setdttm[] PROGMEM =  "日時";
const char S_useoutsr[] PROGMEM =  "日射使用";
const char S_insidesr[] PROGMEM =  "日射量";
const char S_prenttime[] PROGMEM =  "前回自動給液";
const char S_intsrnow[] PROGMEM =  "現在積算日射";
const char S_intsrtd[] PROGMEM =  "本日積算日射";
const char S_ntsuptd[] PROGMEM =  "本日給液量";
const char S_ntsupyd[] PROGMEM =  "昨日給液量";
const char S_amntsup[] PROGMEM =  "総給液量";

//Select Strings
const char S_off[] PROGMEM = "停止";
const char S_on[] PROGMEM = "供給";
const char S_auto[] PROGMEM = "自動";
const char S_stop[] PROGMEM = "強制停止";
const char S_force[] PROGMEM = "強制給液";
const char S_reset[] PROGMEM = "リセット";
const char S_remoteC[] PROGMEM = "CPU遠隔";
const char S_remoteS[] PROGMEM = "手動SW";
const char S_web[] PROGMEM = "下記動作";
const char S_ins[] PROGMEM = "内部設定";
const char S_in[] PROGMEM = "内部";
const char S_out[] PROGMEM = "外部";
const char S_outs[] PROGMEM = "外部設定";
const char *SS_SUP[2]={ S_off, S_on };
const char *SS_MDDP[4]={ S_auto, S_remoteC, S_remoteS, S_web };
const char *SS_MDSET[4]={ S_auto, S_stop, S_force, S_reset };
const char *SS_ISO[4]={ S_in, S_ins, S_out, S_outs };
const char *SS_IO[2]={ S_in, S_out };

//Unit Strings
const char U_perlm[] PROGMEM =  "mℓ/株/分";
const char U_perl[] PROGMEM =  "ℓ/株";
const char U_seci[] PROGMEM =  "秒間";
const char U_minaf[] PROGMEM =  "分後";
const char U_eng[] PROGMEM =  "kJ/m2";
const char U_dpereng[] PROGMEM =  "kJ/m2/日";
const char U_flux[] PROGMEM =  "kW/m2";
const char U_time[] PROGMEM =  "時.分";
const char U_sec[] PROGMEM =  "秒";

//Detail Strings
const char D_1time[] PROGMEM =   "1回で";
const char D_sup1time[] PROGMEM =   "で給液";
const char D_fb2400[] PROGMEM =   "24.00で禁止";
const char D_zero[] PROGMEM =   "0は禁止";
const char D_prev[] PROGMEM =   "終了時刻";

// Dummy Strings
const char EMPTY[] PROGMEM= "";
const char** DUMMY = NULL;

const int U_HtmlLine = 19; //Total number of HTML table rows.
struct UECSUserHtml U_html[U_HtmlLine]={
//{名前,          入出力形式,     単位,       詳細説明,     選択肢文字列,選択肢数, 値,             最小値,最大値, 小数桁数}
  {S_supply,      UECSSHOWSTRING, EMPTY,      EMPTY,        SS_SUP,      2,        &(W_supply),    0,     1,             0},
  {S_modedp,      UECSSHOWSTRING, EMPTY,      EMPTY,        SS_MDDP,     4,        &(W_mode),      0,     3,             0},
  {S_modeset,     UECSSELECTDATA, EMPTY,      EMPTY,        SS_MDSET,    4,        &(W_modeset),   0,     3,             0},
  {S_time,        UECSINPUTDATA,  U_seci,     D_1time,      DUMMY,       0,        &(W_onesp),     1,     1200,          0},
  {S_insidesr,    UECSSHOWDATA,   U_flux,     EMPTY,        DUMMY,       0,        &(W_sr_in),     0,     0,             3},
  {S_useoutsr,    UECSSELECTDATA, EMPTY,      EMPTY,        SS_IO,       2,        &(W_useoutsrf), 0,     1,             0},
  {S_intsrnow,    UECSSHOWDATA,   U_eng,      EMPTY,        DUMMY,       0,        &(W_srint_pr),  0,     0,             3},
  {S_spintsr,     UECSINPUTDATA,  U_eng,      D_sup1time,   DUMMY,       0,        &(W_ntspsrint), 1,     5000,          0},
  {S_intsrtd,     UECSSHOWDATA,   U_eng,      EMPTY,        DUMMY,       0,        &(W_srint_td),  0,     0,             3},
  {S_insidetm,    UECSINPUTDATA,  U_time,     EMPTY,        DUMMY,       0,        &(W_time_in),   0,     2359,          2},
  {S_setdttm,     UECSSELECTDATA, EMPTY,      EMPTY,        SS_ISO,      4,        &(W_setdtf),    0,     3,             0},
  {S_fcntsup1,    UECSINPUTDATA,  U_time,     D_fb2400,     DUMMY,       0,        &(W_fctime1),   0,     2400,          2},
  {S_fcntsup2,    UECSINPUTDATA,  U_time,     D_fb2400,     DUMMY,       0,        &(W_fctime2),   0,     2400,          2},
  {S_prenttime,   UECSSHOWDATA,   U_time,     D_prev,       DUMMY,       0,        &(W_pnrtime),   0,     0,             2},
  {S_intfcntsup,  UECSINPUTDATA,  U_minaf,    D_zero,       DUMMY,       0,        &(W_ntfcspint), 0,     720,           0},
  {S_const,       UECSINPUTDATA,  U_perlm,    EMPTY,        DUMMY,       0,        &(W_ntconst),   0,     1000,          1},  
  {S_ntsuptd,     UECSSHOWDATA,   U_perl,     EMPTY,        DUMMY,       0,        &(W_amnt_td),   0,     0,             3},
  {S_ntsupyd,     UECSSHOWDATA,   U_perl,     EMPTY,        DUMMY,       0,        &(W_amnt_yd),   0,     0,             3},
  {S_amntsup,     UECSSHOWDATA,   U_perl,     EMPTY,        DUMMY,       0,        &(W_amnt_all),  0,     0,             3},
};

//////////////////////////////////
// UserCCM setting
//////////////////////////////////

//define CCMID for identify
//CCMID_dummy must put on last
enum { 
  CCMID_cnd,      CCMID_rcM,    CCMID_rcA,
  CCMID_TdNtInt,  CCMID_SrInt,  CCMID_WRadiation,
  CCMID_Time,
  CCMID_dummy
};
const int U_MAX_CCM = CCMID_dummy;
UECSCCM U_ccmList[U_MAX_CCM];

const char ccmNameCnd[] PROGMEM = "機器状態";
const char ccmTypeCnd[] PROGMEM = "cnd.cNM";
const char ccmUnitCnd[] PROGMEM = "";

const char ccmNamercA[] PROGMEM = "遠隔制御";
const char ccmTypercA[] PROGMEM = "rcA.cNM";

const char ccmNamercM[] PROGMEM = "手動操作";
const char ccmTypercM[] PROGMEM = "rcM.cNM";
const char ccmUnitP[] PROGMEM = "%";

const char ccmNameTdNtInt[] PROGMEM = "日給液量";
const char ccmTypeTdNtInt[] PROGMEM = "TdNtInt.cNM";
const char ccmUnitTdNtInt[] PROGMEM = "L plant-1";

const char ccmNameSrInt[] PROGMEM = "日射積算";
const char ccmTypeSrInt[] PROGMEM = "SrInt.cNM";
const char ccmUnitSrInt[] PROGMEM = "kJ";

const char ccmNameWRad[] PROGMEM = "日射フラックス";
const char ccmTypeWRad[] PROGMEM = "WRadiation.mOC";
const char ccmUnitRad[] PROGMEM = "kW m-2";

const char ccmNameTime[] PROGMEM = "時刻";
const char ccmTypeTime[] PROGMEM = "Time";
const char ccmUnitTime[] PROGMEM = "時分";

//------------------------------------------------------
//UARDECS初期化用関数
//主にCCMの作成とMACアドレスの設定を行う
//------------------------------------------------------
void UserInit(){

 //Set ccm list
 UECSsetCCM(true,  CCMID_cnd,        ccmNameCnd,       ccmTypeCnd,      ccmUnitCnd,      30, 0, A_1S_0 );
 UECSsetCCM(false, CCMID_rcA,        ccmNamercA,       ccmTypercA,      ccmUnitP,        30, 0, A_1M_0 );
 UECSsetCCM(false, CCMID_rcM,        ccmNamercM,       ccmTypercM,      ccmUnitP,        30, 0, A_1S_0 );
 UECSsetCCM(true,  CCMID_TdNtInt,    ccmNameTdNtInt,   ccmTypeTdNtInt,  ccmUnitTdNtInt,  30, 3, A_1M_0 );
 UECSsetCCM(true,  CCMID_SrInt,      ccmNameSrInt,     ccmTypeSrInt,    ccmUnitSrInt,    30, 3, A_1M_0 );
 UECSsetCCM(false, CCMID_WRadiation, ccmNameWRad,      ccmTypeWRad,     ccmUnitRad,      30, 3, A_1M_0 );
 UECSsetCCM(false, CCMID_Time,       ccmNameTime,      ccmTypeTime,     ccmUnitTime,     30, 0, A_1M_0 );


 //Set proper MAC address
 U_orgAttribute.mac[0] = ma[0] ;
 U_orgAttribute.mac[1] = ma[1];
 U_orgAttribute.mac[2] = ma[2];
 U_orgAttribute.mac[3] = ma[3];
 U_orgAttribute.mac[4] = ma[4];
 U_orgAttribute.mac[5] = ma[5];

}

//---------------------------------------------------------
//起動直後に１回呼び出される関数。
//様々な初期化処理を記述できる。
//この関数内ではUECSsetup()関数を呼び出さなくてはならない。
//必要に応じて処理を記述してもかまわない。
//---------------------------------------------------------
void setup(){

  //制御シールドの制御信号線の初期化
  pinMode(in_out, OUTPUT) ;
  digitalWrite(in_out, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], OUTPUT) ;
    digitalWrite(port[i], LOW) ;
  }
  pinMode(ratch, OUTPUT) ;
  digitalWrite(ratch, HIGH) ;
  delayMicroseconds(1) ; 
  digitalWrite(ratch, LOW) ;
  pinMode(wdt, OUTPUT) ;
  
  digitalWrite(wdt, HIGH) ; // ウオッチドッグクリア信号を初期値にする
  clearWdt() ;
  clearDout() ; // 出力をリセット

  // I2C初期化
  Wire.begin();
  
  // RTC初期化 付いていても、いなくても、とりあえずリセット
  // ここでハングアップしないので
  Wire.beginTransmission(RTC_ADRS);
  Wire.write(0xE0);
  Wire.write(0x20);  //24hourMode
  Wire.write(0x00);  // PON Clear
  Wire.endTransmission();
  delay(1);

  UECSsetup();
  
  // W_mode変数の異常でリセットの条件判断
  if((W_mode<0)||(W_mode>2)) {
    W_modeset = 3 ; //全値リセットに設定
    initializeSetpoints();
    }

  //時計設定により時刻初期値
  if(W_setdtf == 2) {
    // 外部なので時刻リセット
    v_time = W_time_in = 0 ; // 時刻0時0分
    minutes = 0 ; // 分刻0分
  }
  if(W_setdtf == 0) {
    // 内部なので
    // リアルタイムクロックから時刻読み込み
    RTCread() ;
    // RTCの値が変ならばリセット
    if((v_time > 2359)||(v_time < 0)) {
      v_time = 0 ;
      v_date = 170101 ;
      RTCset() ;
    }
  }
  pminutes = minutes ; // 一回前の分を同値にクリア
  
  //自動給液残秒数リセット
  rNutSec = 0 ;
  //1分間分の積算変数クリア
  W_srint_pr = 0 ;
  //前回給液時刻を現在時刻にセット(0にすると起動時にほぼ給液されてしまう)
  W_pnrtime = v_time ;
  // 秒カウンタをリセット
  Csec = 0 ;
  //積算給液量計算用作業変数クリア
  intsupply = 0 ;

}

//------------------------------------------------------
// EPROM関係の制御設定値等をクリアする。
// W_modeの変数値がおかしい場合もリセット
//------------------------------------------------------
void initializeSetpoints() {
  if(W_modeset == 3) {
    W_supply = 0 ; // 培養液供給
    W_mode = 3 ; // 給液モード 下記強制動作
    W_modeset = 1 ; // 給液モード設定 強制停止
    W_ntconst = 100 ; // 給液定数 10.0 ml
    W_onesp = 120 ; // 給液時間 120秒
    W_ntspsrint = 2000 ; // 給液積算日射
    W_ntfcspint = 240 ; // 強制給液間隔
    W_fctime1 = 2400 ; // 強制給液時刻1
    W_fctime2 = 2400 ; // 強制給液時刻2
    W_setdtf = 2 ; // 日時 外部使用
    W_useoutsrf = 1 ; // 外部日射使用
    W_sr_in = 0 ; // 内部日射量
    W_srint_td = 0 ; // 本日積算日射
    W_pnrtime = v_time ; // 前回給液時刻
    W_srint_pr = 0 ; // 現在積算日射
    W_amnt_td = 0 ; // 本日給液量
    W_amnt_yd = 0 ; // 昨日給液量
    W_amnt_all = 0 ; // 総給液量
  }
}


//------------------------------------------------------
// 手動操作が指定されたときに所定の動作を実施する
//------------------------------------------------------
void manualOperation() {
  if((W_modeset == 0)&&(W_mode == 3)) {
    // 自動になった場合
    W_mode = 0 ; // 強制モード
    nutrientOFF() ;
  }
  if(W_modeset == 1) {
    // 強制給液停止の場合
    W_mode = 3 ; // 強制モード
    nutrientOFF() ;
  }
  if(W_modeset == 2) {
    // 強制給液の場合
    W_mode = 3 ; // 強制モード
    nutrientON() ;
  }
}

//------------------------------------------------------
// ネットからのリモート操作 rcM.cMNのCCM
// 有効のとき、-1: リモート操作解除、
//             0 給液停止
//             1-100　給液開始
//------------------------------------------------------
void rcMOperation() {  
  if(W_mode <= 2) {
    // auto,rcA,rcMの時のみ処理
    if(!(U_ccmList[CCMID_rcM].validity)) {
      if(W_mode == 2) {
        W_mode = 0 ; //AUTOモード
        nutrientOFF() ;
      }
    }
    // リモートCCMが有効だったら処理
    else {
      // rcMが-1でリモート操作モードだったら解除
      if(U_ccmList[CCMID_rcM].value == -1) {
        if(W_mode == 2) {
          W_mode = 0 ; //AUTOモード
          nutrientOFF() ;
        }
      }
      else {
        // rcMが0-100%の意味ある値であれば
        if((U_ccmList[CCMID_rcM].value >= 0)&&(U_ccmList[CCMID_rcM].value <= 100)) {
          W_mode = 2 ; // リモート操作モードに
          // それぞれの値で処理
          switch(U_ccmList[CCMID_rcM].value) {
            case 0:
              nutrientOFF() ;
              break ;
            default: // それ以外、つまり1-100のとき
              nutrientON() ;
              break ;
          }
        }
      }
    }
  }
}

//------------------------------------------------------
// ネットからのリモート制御 rcA.cMNのCCM
// 有効のとき、-1: リモート制御解除、
//             0 給液停止
//             1-100　給液開始
//------------------------------------------------------
void rcAOperation() {
  if(W_mode <= 1) {
    // auto,rcAの時のみ処理
    if(!(U_ccmList[CCMID_rcA].validity)) {
      if(W_mode == 1) {
        W_mode = 0 ; //AUTOモード
        nutrientOFF() ;
      }
    }
    // リモートCCMが有効だったら処理
    else {
      // rcAが-1でリモート制御モードだったら解除
      if(U_ccmList[CCMID_rcA].value == -1) {
        if(W_mode == 1) {
          W_mode = 0 ; //AUTOモード
          nutrientOFF() ;
        }
      }
      else {
        // rcAが0-100%の意味ある値であれば
        if((U_ccmList[CCMID_rcA].value >= 0)&&(U_ccmList[CCMID_rcA].value <= 100)) {
          W_mode = 1 ; // リモート制御モードに
          // それぞれの値で処理
          switch(U_ccmList[CCMID_rcA].value) {
            case 0:
              nutrientOFF() ;
              break ;
            default: // それ以外、つまり1-100のとき
              nutrientON() ;
              break ;
          }
        }
      }
    }
  }
}

//------------------------------------------------------
// 自動給液の処理プログラム
//------------------------------------------------------
// 1分毎に呼びだして条件をチェックし給液時間をrNutSecに加算
void autoCheck() { 
  if(W_mode == 0) {
    // 自律モードであれば
    // 強制時刻での給液
    if(v_time == W_fctime1) {
      rNutSec += W_onesp ;
      W_srint_pr = 0 ; // 積算日射クリア
    }
    if(v_time == W_fctime2) {
      rNutSec += W_onesp ;
      W_srint_pr = 0 ; // 積算日射クリア
    }
    //強制給液間隔超過による強制給液
    if((W_ntfcspint > 0)&&(W_sr_in > 1)) { // 設定値が0の時または日射量が0.001kW以下のときには行わない
      if(W_pnrtime == 0) W_pnrtime = v_time ; // 午前0時から初めて日射量が0でなくなった時には、
      //ここから強制給液経過時間を計算し、夜明け時に強制給液しないために、前回給液時刻を夜明けの時刻に設定する
      if(elapsMin(v_time, W_pnrtime) >= W_ntfcspint) {
        rNutSec += W_onesp ;
        W_srint_pr = 0 ; // 積算日射クリア
      }
    }
    //積算日射超過による自動給液
    if(W_srint_pr >= (W_ntspsrint * 1000)) {
       rNutSec += W_onesp ;
       W_srint_pr = 0 ;
    }
  }
}

// 1秒毎に呼びだしてrNutSecで給液し、1減算
void autoControl(){
  if(W_mode == 0) {
    //自律給液動作
    if(rNutSec > 0) {
      nutrientON() ;
      W_pnrtime = v_time ;
    }
    else nutrientOFF() ;
  } 
  if(rNutSec > 0) rNutSec-- ; // 養液供給残時間の減算
}  
  
//------------------------------------------------------
// 日射量培養液午前0時関係計算
//------------------------------------------------------
void calc0am(){
  //日射量関係
  W_srint_td = 0 ;
  W_srint_pr = 0 ;
  
  //培養液関係
  W_amnt_yd = W_amnt_td ;
  W_amnt_td = 0 ;
  
  //前回給液時刻もリセットする
  W_pnrtime = 0 ;
}  

//------------------------------------------------------
// CCMに値をセットする。
// 毎秒チェック
//------------------------------------------------------
void setCCM(){
  // cndに制御モード値をセット
  U_ccmList[CCMID_cnd].value = W_mode ;
  // 本日給液量セット
  U_ccmList[CCMID_TdNtInt].value = W_amnt_td ;
  // 本日日射積算値セット
  U_ccmList[CCMID_SrInt].value = W_srint_td ;
}

//---------------------------------------------------------
//Webページから入力が行われ各種値を設定し不揮発性メモリに値を保存後、
//ページの再描画を行う前に以下の関数が呼び出される。
//---------------------------------------------------------
void OnWebFormRecieved(){
  if(W_setdtf == 3) { // 外部設定ならば、時刻をセットしてから外部読み込み
    v_time = W_time_in ;
    W_setdtf = 2 ;
    W_pnrtime = v_time ; // 時刻変更後強制給液しないために前回給液時刻にもセット
    Csec = 0 ; // 秒もリセット
  }

  if(W_setdtf == 2) { // 外部ならば、とりあえず外部時刻を読んでみようとする
    if(U_ccmList[CCMID_Time].validity) { // Time CCMが有効ならば
      W_time_in = v_time = U_ccmList[CCMID_Time].value / 100 ; // 読み込みセット
    }
  }

  if(W_setdtf == 0) { // 内部時計使用ならば
    // リアルタイムクロックから時刻読み込み
    RTCread() ;
    W_time_in = v_time ;
  }
  
  if(W_setdtf == 1) { // 内部時計の設定ならば
    // 内部時刻変更
    v_time = W_time_in ;
    v_date = 170101 ;
    RTCset() ;
    W_setdtf = 0 ;
    W_pnrtime = v_time ; // 時刻変更後強制給液しないために前回給液時刻にもセット
    Csec = 0 ; // 秒もリセット
  }

  // モード変更に基づく処理　飛び先に条件判断あり
  initializeSetpoints(); //リセット処理
  manualOperation(); // Web強制操作処理

}

//---------------------------------------------------------
//１秒に１回呼び出される関数
//---------------------------------------------------------
void UserEverySecond(){

  //ウオッチドッグタイマークリア
  clearWdt() ;
  
  //時刻関係処理
  pminutes = minutes ; // 1分の変わり目の判断変数の更新
  // 現在時刻読み込みか計算
  if(W_setdtf == 2) { // 外部ならば、とりあえず外部時刻を読んでみようとする
    Csec++ ; // 秒カウンタUP
    if(U_ccmList[CCMID_Time].validity) { // Time CCMが有効ならば
      W_time_in = v_time = U_ccmList[CCMID_Time].value / 100 ; // 読み込みセット
      if(Csec > 59) Csec = 0 ;
    }
    else { // CCM無効のときには不正確だが自律的に経時
      if(Csec > 59) {
        Csec = 0 ;
        if(v_time > 2359) { v_time = 0 ; }
        else {
          if((v_time % 100) == 59) { v_time = v_time + 100 - 59 ; }
          else v_time++ ;
        }
        W_time_in = v_time ;
      }
    }
    minutes = v_time % 100 ; // minutesに分刻をセット
  }
  if(W_setdtf == 0) { // 内部ならば、
    RTCread() ; //v_timeに時刻読み出し
    W_time_in = v_time ;
    Csec++ ; // RTCの時にはminutesがRTCから読み込み時に自動セットされるが
             // 外部と内部の切替時に不安定にならないように秒カウンタを処理
    if(Csec > 59) Csec = 0 ;
  }

  if(minutes != pminutes) {
    //////////////////////////////////////////////////////
    // 1分毎の処理
    // void UserEveryMinute()は誤差がある可能性があるので
    // 使用しない(1回給液しなかったりすると大変!)
    //////////////////////////////////////////////////////
    Csec = 0 ; // 秒カウンタ強制リセット
    
    // 0:00の処理
    if(v_time == 0) calc0am() ; // 日射量培養液午前0時関係計算
    autoCheck() ; // 1分毎の自動給液条件チェック    
    W_amnt_td += (intsupply / 600) ; // 本日給液量の計算
    W_amnt_all += (intsupply / 600) ; // 総給液量の計算
    intsupply = 0 ; // 1分間積算値クリア
  }
  
  /////////////////////
  // 以下は1秒毎の処理
  /////////////////////
  autoControl() ; // 自律自動制御処理
  rcMOperation() ; // リモート操作処理
  rcAOperation() ; // リモート制御処理
  
  if(W_useoutsrf == 1) { // 外部日射を使用の設定
    if(U_ccmList[CCMID_WRadiation].validity) {
      // 外部日射を使用の設定で、CCMが有効ならば、外部値を積算
      W_sr_in = U_ccmList[CCMID_WRadiation].value ;
      W_srint_pr += (float)U_ccmList[CCMID_WRadiation].value ;
      W_srint_td += (float)U_ccmList[CCMID_WRadiation].value ;
    }
    else { // CCMが無効ならば、0}
      W_sr_in = 0 ;
    }
  }
  else {
    // それ以外は内部日射を積算
    W_sr_in = (signed long)SRmeasure() ; // センサ日射量測定
    W_srint_pr += W_sr_in ;
    W_srint_td += W_sr_in ;
  }
  
  //供給培養液量計算
  if(W_supply == 1) intsupply += W_ntconst ;
  
  setCCM() ; // CCMの値計算
}

//---------------------------------------------------------
//１分に１回呼び出される関数
//---------------------------------------------------------
void UserEveryMinute(){
}

//---------------------------------------------------------
//メインループ
//システムのタイマカウント，httpサーバーの処理，
//UDP16520番ポートと16529番ポートの通信文をチェックした後，呼び出さされる関数。
//呼び出される頻度が高いため，重い処理を記述しないこと。
//---------------------------------------------------------
void UserEveryLoop(){
}

//---------------------------------------------------------
//setup()実行後に呼び出されるメインループ
//この関数内ではUECSloop()関数を呼び出さなくてはならない。
//UserEveryLoop()に似ているがネットワーク関係の処理を行う前に呼び出される。
//必要に応じて処理を記述してもかまわない。
//呼び出される頻度が高いため,重い処理を記述しないこと。
//---------------------------------------------------------
void loop(){
  UECSloop();
}

