/*
 Low-cost UECS roof ventilator node
 LU-RFVENT By T.Hoshi, Kindai Univ.
 2017.12.14 version 1.0
*/

// このプログラムは、無償提供する代わりに、動作結果に関する
// 一切の保証は致しません。自己責任でお使いください。また、
// 改造も自由ですが、公刊等をされる場合は、「近畿大学　星　
// 岳彦が開発した○○〇プログラムを用いた」と引用して下さい
// ますと、今後の研究開発の励みになりますので、どうぞよろ
// しくお願いします。　　　　　2019.3.4 Takehiko Hoshi
// 
// このシールドはArduino MEGA 2560 + Ethernet Shiedld2の
// 組み合わせで動作させることを前提にしています。
// プログラムを書き込むときには、Arduino MEGA 2560だけ
// の状態で書き込んでください。
// Ethernet Sheild 2基板のシールのmacアドレスに変更して
// ください。このプログラムをmacで検索すると付近にあり
// ます。macアドレスとは。90:A2:DA:0F:xx:xx のような8個
// の16進数です。これをばらして0xの後に転記してください。

// 出力の割り付け状態
// Dout0 天窓　開信号
// Dout1 天窓  閉信号
// Dout2 なし
// Dout3 なし
// Din0 降水センサー　電圧印加で降水信号

#include <SPI.h>
#include <Ethernet2.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <EEPROM.h>
// UECS用のライブラリUARDECSのインストール
// https://uecs.org/arduino/uardecs.html が必要です。
#include <Uardecs.h>

//////////////////////
// 定数関連定義
//////////////////////

//制御出力の定義
const uint8_t openVT = 0 ; // 窓開け出力
const uint8_t closeVT = 1 ; // 窓閉め出力
const uint8_t rain = 0 ; // 降雨信号

//
//IPアドレスをリセットするピンの割付
//
//Pin ID. This pin is pull-upped automatically.
const byte U_InitPin = 57 ; // for Arduino MEGA 2560 
//const byte U_InitPin = 17 ; // for Arduino Ethernet
//This shields is assigned as A3 for reset pin.
//When U_InitPin status equals this value,IP address is set "192.168.1.7".
//(This is added Ver3 over)
const byte U_InitPin_Sense= HIGH ;  

/////////////////////////////
// ノードの基礎情報の設定
/////////////////////////////

const char U_name[] PROGMEM =   "LU-RFVENT";
const char U_vender[] PROGMEM = "Hoshi-lab.";
const char U_uecsid[] PROGMEM = "011009001004";
const char U_footnote[] PROGMEM = "UARDECS7.5 Product by <A HREF=\"http://hoshi-lab.info/\">Kindai, hoshi-lab.</A>";
char U_nodename[20] = "LU-RFVENT";
UECSOriginalAttribute U_orgAttribute;

////////////////////////////////////////////
// CO2HEATノードの変数設定とライブラリ
////////////////////////////////////////////

// 入出力ポートの定義
const int port[4] = { 7, 8, 5, 6 } ;
// ウオッチドッグタイマーのリセットポート
const int wdt = 9 ; // 0 = clear timer
// ディジタル出力値の一時記憶装置のトリガポート
const int ratch = 3 ; //  0->1 ratch
// ディジタル入力値のイネーブルポート
const int in_out = 2 ; // in = 1 out =0

//制御関係定数・変数
int openOpsec ; // 開動作する秒数　(セットするとプラスの時に開動作)
                // 1秒ごとに1減算　－3600でそれ以上減算しない
int closeOpsec ; // 閉動作する秒数　(セットするとプラスの時に閉動作)
                 // 1秒ごとに1減算　－3600でそれ以上減算しない
int innerTime ; // 内部時刻hhmm
int secCount ; // 60秒に一回の処理
int openMotor, closeMotor ; // 各運転状況フラグ HIGHで運転
float ventOpenPerOrg ; // 天窓開度を入れるオリジナル値

// 作業変数
int i ; // ループ用変数
int DOpin[4] ; // 制御出力配列
int DIpin[4] ; // ディジタル入力配列

//Define variables for Web interface
signed long W_RVcond ; // 窓開閉状態
signed long W_RVmode ; // モード
signed long W_RCopCond ; // 動作設定
signed long W_tempSP1 ; // 換気気温1
signed long W_timeSP2 ; // 開始時刻2
signed long W_tempSP2 ; // 換気気温2
signed long W_timeSP3 ; // 開始時刻3
signed long W_tempSP3 ; // 換気気温3
signed long W_timeSP4 ; // 開始時刻4
signed long W_tempSP4 ; // 換気気温4
signed long W_TempHis ; // 気温幅±
signed long W_FopenSec ; // 全開所要時間
signed long W_FcloseSec ; // 全閉所要時間
signed long W_percent1 ; // 気温差感度
signed long W_TDifAdj ; // 気温差感度
// 単位: ℃/10%
// 内外気温差が1℃以下の時: 1回で100%開閉
// 内外気温差がTDifAdj以上の時: 1回で10%開閉
// その範囲内の時には、INT(100-(Ti-To-1)/TDiAdj*90/10)*10
// 1回1分間隔 10%単位に切り捨て

// ウオッチドッグタイマーのクリア(2秒以内に1回の実行が必要)
void clearWdt() {
  digitalWrite(wdt, LOW) ;
  delayMicroseconds(100) ;
  digitalWrite(wdt, HIGH) ;
}

// ディジタル値を出力する d[0]-d[3]の値 0はOFF、1はON
void dout(int *d) {
  digitalWrite(in_out, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], OUTPUT) ;
    digitalWrite(port[i], d[i]) ;
  }
  delayMicroseconds(1) ;
  digitalWrite(ratch, HIGH) ;
  delayMicroseconds(1) ; 
  digitalWrite(ratch, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], INPUT) ;
  }
  digitalWrite(in_out, HIGH) ;
}

// 制御出力を全てOFFにする
void clearDout() {
  int i;
  for(i=0; i<4; i++) DOpin[i] = LOW ;
  dout(DOpin) ;
  openMotor = closeMotor = LOW ;
}

// フラグに従い出力
void controlOut() { // 両方ONの時はリセット
  if((openMotor == HIGH)&&(closeMotor == HIGH)) {
    openMotor = closeMotor = LOW ;
    return ;
  }
  DOpin[openVT] = openMotor ;
  DOpin[closeVT] = closeMotor ;
  dout(DOpin) ;
}
 
// ディジタル入力値の読み込み d[0]-d[3]に値を代入
void din() {
  digitalWrite(in_out, HIGH) ;
  delayMicroseconds(1) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], INPUT) ;
    delayMicroseconds(1) ;
    DIpin[i] = digitalRead(port[i]) ;
  }
}

//////////////////////////////////
// html page setting
//////////////////////////////////

// Define Strings
const char S_RVcond[] PROGMEM ="窓開閉状態" ;
const char S_RVmode[] PROGMEM ="制御モード" ;
const char S_RCopCOnd[] PROGMEM ="動作設定" ;
const char S_tempSP1[] PROGMEM ="換気気温1" ;
const char S_timeSP2[] PROGMEM ="開始時刻2" ;
const char S_tempSP2[] PROGMEM ="換気気温2" ;
const char S_timeSP3[] PROGMEM ="開始時刻3" ;
const char S_tempSP3[] PROGMEM ="換気気温3" ;
const char S_timeSP4[] PROGMEM ="開始時刻4" ;
const char S_tempSP4[] PROGMEM ="換気気温4" ;
const char S_TempHis[] PROGMEM ="気温幅±" ;
const char S_FopenSec[] PROGMEM ="全開時間" ;
const char S_FcloseSec[] PROGMEM ="全閉時間" ;
const char S_percent1[] PROGMEM ="1回開閉量" ;
const char S_TDifAdj[] PROGMEM ="気温差感度" ;

//Select Strings
const char S_stop[] PROGMEM = "停止";
const char S_open[] PROGMEM = "開動作";
const char S_close[] PROGMEM = "閉動作";
const char S_auto[] PROGMEM = "自動";
const char S_manual[] PROGMEM = "Web操作";
const char S_rCont[] PROGMEM = "遠隔制御";
const char S_rOp[] PROGMEM = "遠隔操作";
const char S_Fopen[] PROGMEM = "強制開";
const char S_Fclose[] PROGMEM = "強制閉";
const char S_reset[] PROGMEM = "初期化";
const char *SS_RVcond[3]={ S_stop, S_open, S_close };
const char *SS_RVmode[5]={ S_stop, S_auto, S_manual, S_rCont, S_rOp };
const char *SS_RCopCond[5]={ S_stop, S_auto, S_Fopen, S_Fclose, S_reset };

//Unit Strings
const char U_sec[] PROGMEM =  "秒";
const char U_temp[] PROGMEM =  "℃";
const char U_time[] PROGMEM =  "時.分";
const char U_percent1[] PROGMEM =  "%/回";
const char U_TDifAdj[] PROGMEM =  "℃/10%";

//Detail Strings
const char D_init[] PROGMEM =   "初期値";
const char D_fb2400[] PROGMEM =   "24.00無視";
const char D_hist[] PROGMEM =   "不感帯";
const char D_ReqT[] PROGMEM =   "所要時間";
const char D_TDifAdj[] PROGMEM =   "10%/回の気温差";

// Dummy Strings
const char EMPTY[] PROGMEM= "";
const char** DUMMY = NULL;

const int U_HtmlLine = 15; //Total number of HTML table rows.

struct UECSUserHtml U_html[U_HtmlLine] = {
//{  名前,          入出力形式,     単位,  詳細説明,   選択肢文字列,選択肢数, 値,               最小値,最大値, 小数桁数}
  { S_RVcond,       UECSSHOWSTRING, EMPTY,  EMPTY,     SS_RVcond,   3,        &(W_RVcond),      0,     2,      0},
  { S_RVmode,       UECSSHOWSTRING, EMPTY,  EMPTY,     SS_RVmode,   5,        &(W_RVmode),      0,     4,      0},
  { S_RCopCOnd,     UECSSELECTDATA, EMPTY,  EMPTY,     SS_RCopCond, 5,        &(W_RCopCond),    0,     4,      0},
  { S_tempSP1,      UECSINPUTDATA,  U_temp, D_init,    DUMMY,       0,        &(W_tempSP1),     0,     40,     0},
  { S_timeSP2,      UECSINPUTDATA,  U_time, D_fb2400,  DUMMY,       0,        &(W_timeSP2),     0,     2400,   2},
  { S_tempSP2,      UECSINPUTDATA,  U_temp, EMPTY,     DUMMY,       0,        &(W_tempSP2),     0,     40,     0},
  { S_timeSP3,      UECSINPUTDATA,  U_time, D_fb2400,  DUMMY,       0,        &(W_timeSP3),     0,     2400,   2},
  { S_tempSP3,      UECSINPUTDATA,  U_temp, EMPTY,     DUMMY,       0,        &(W_tempSP3),     0,     40,     0},
  { S_timeSP4,      UECSINPUTDATA,  U_time, D_fb2400,  DUMMY,       0,        &(W_timeSP4),     0,     2400,   2},
  { S_tempSP4,      UECSINPUTDATA,  U_temp, EMPTY,     DUMMY,       0,        &(W_tempSP4),     0,     40,     0},
  { S_TempHis,      UECSINPUTDATA,  U_temp, D_hist,    DUMMY,       0,        &(W_TempHis),     0,     10,     0},
  { S_FopenSec,     UECSINPUTDATA,  U_sec,  D_ReqT,    DUMMY,       0,        &(W_FopenSec),    0,     1800,   0},
  { S_FcloseSec,    UECSINPUTDATA,  U_sec,  D_ReqT,    DUMMY,       0,        &(W_FcloseSec),   0,     1800,   0},
  { S_percent1,     UECSSHOWDATA,   U_percent1, EMPTY, DUMMY,       0,        &(W_percent1),    0,     100,    0},
  { S_TDifAdj,      UECSINPUTDATA,  U_temp, D_TDifAdj, DUMMY,       0,        &(W_TDifAdj),     0,     40,     0}
} ;

//////////////////////////////////
// UserCCM setting
//////////////////////////////////

//define CCMID for identify
//CCMID_dummy must put on last
enum { 
  CCMID_InAirTemp_mIC, CCMID_WAirTemp_mOC,   CCMID_Time,
  CCMID_rcM_cRV,       CCMID_rcA_cRV,        CCMID_cnd_cRV,
  CCMID_OpenP_cRV,     CCMID_VentSP_cRV,     CCMID_RainS_cRV,
  CCMID_esec_cRV,      CCMID_dummy
};
const int U_MAX_CCM = CCMID_dummy;
UECSCCM U_ccmList[U_MAX_CCM];

const char ccmNameInAirTemp[] PROGMEM = "室内気温";
const char ccmTypeInAirTemp[] PROGMEM = "InAirTemp.mIC";
const char ccmUnitInAirTemp[] PROGMEM = "C";

const char ccmNameWAirTemp[] PROGMEM = "室外気温";
const char ccmTypeWAirTemp[] PROGMEM = "WAirTemp.mOC";
const char ccmUnitWAirTemp[] PROGMEM = "C";

const char ccmNameTime[] PROGMEM = "時刻";
const char ccmTypeTime[] PROGMEM = "Time";
const char ccmUnitTime[] PROGMEM = "hhmmss";

const char ccmNamercM[] PROGMEM = "遠隔操作";
const char ccmTypercM[] PROGMEM = "rcM.cRV";
const char ccmUnitrcM[] PROGMEM = "%";

const char ccmNamercA[] PROGMEM = "遠隔制御";
const char ccmTypercA[] PROGMEM = "rcA.cRV";
const char ccmUnitrcA[] PROGMEM = "%";

const char ccmNamecnd[] PROGMEM = "状態";
const char ccmTypecnd[] PROGMEM = "cnd.cRV";
const char ccmUnitcnd[] PROGMEM = "";

const char ccmNameOpenP[] PROGMEM = "天窓開度";
const char ccmTypeOpenP[] PROGMEM = "OpenP.cRV";
const char ccmUnitOpenP[] PROGMEM = "%";

const char ccmNameVentSP[] PROGMEM = "換気気温";
const char ccmTypeVentSP[] PROGMEM = "VentSP.cRV";
const char ccmUnitVentSP[] PROGMEM = "C";

const char ccmNameRainS[] PROGMEM = "感雨";
const char ccmTypeRainS[] PROGMEM = "RainS.cRV";
const char ccmUnitRainS[] PROGMEM = "";

// WDT動作確認用
const char ccmNameEsec[] PROGMEM = "経過時間";
const char ccmTypeEsec[] PROGMEM = "esec.cRV";
const char ccmUnitsec[] PROGMEM = "s";

//------------------------------------------------------
//UARDECS初期化用関数
//主にCCMの作成とMACアドレスの設定を行う
//------------------------------------------------------
void UserInit(){

 //Set ccm list
 //         send,  CCMID,                Name,              Type Name,         Unit,              Pri, dig, class
 UECSsetCCM(false, CCMID_InAirTemp_mIC,  ccmNameInAirTemp,  ccmTypeInAirTemp,  ccmUnitInAirTemp,  30,  1,   A_10S_0 );
 UECSsetCCM(false, CCMID_WAirTemp_mOC,   ccmNameWAirTemp,   ccmTypeWAirTemp,   ccmUnitWAirTemp,   30,  1,   A_10S_0 );
 UECSsetCCM(false, CCMID_Time,           ccmNameTime,       ccmTypeTime,       ccmUnitTime,       30,  0,   A_1M_0  );
 UECSsetCCM(false, CCMID_rcM_cRV,        ccmNamercM,        ccmTypercM,        ccmUnitrcM,        30,  0,   A_1S_0  );
 UECSsetCCM(false, CCMID_rcA_cRV,        ccmNamercA,        ccmTypercA,        ccmUnitrcA,        30,  0,   A_1M_0  );
 UECSsetCCM(true,  CCMID_cnd_cRV,        ccmNamecnd,        ccmTypecnd,        ccmUnitcnd,        30,  0,   A_1S_0  );
 UECSsetCCM(true,  CCMID_OpenP_cRV,      ccmNameOpenP,      ccmTypeOpenP,      ccmUnitOpenP,      30,  0,   A_10S_0 );
 UECSsetCCM(true,  CCMID_VentSP_cRV,     ccmNameVentSP,     ccmTypeVentSP,     ccmUnitVentSP,     30,  0,   A_1M_0  );
 UECSsetCCM(true,  CCMID_RainS_cRV,      ccmNameRainS,      ccmTypeRainS,      ccmUnitRainS,      30,  0,   A_10S_0 );
 // WDT動作確認用
 UECSsetCCM(true,  CCMID_esec_cRV,       ccmNameEsec,       ccmTypeEsec,       ccmUnitsec,        30,  0,   A_1S_0  );

 //Set proper MAC address
 U_orgAttribute.mac[0] = 0x90;
 U_orgAttribute.mac[1] = 0xA2;
 U_orgAttribute.mac[2] = 0xDA;
 U_orgAttribute.mac[3] = 0x11;
 U_orgAttribute.mac[4] = 0x1D;
 U_orgAttribute.mac[5] = 0x3A;

}

//---------------------------------------------------------
//起動直後に１回呼び出される関数。
//様々な初期化処理を記述できる。
//この関数内ではUECSsetup()関数を呼び出さなくてはならない。
//必要に応じて処理を記述してもかまわない。
//---------------------------------------------------------
void setup(){

  //制御シールドの制御信号線の初期化
  pinMode(in_out, OUTPUT) ;
  digitalWrite(in_out, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], OUTPUT) ;
    digitalWrite(port[i], LOW) ;
  }
  pinMode(ratch, OUTPUT) ;
  digitalWrite(ratch, HIGH) ;
  delayMicroseconds(1) ; 
  digitalWrite(ratch, LOW) ;
  pinMode(wdt, OUTPUT) ;
  
  digitalWrite(wdt, HIGH) ; // ウオッチドッグクリア信号を初期値にする
  clearWdt() ;
  clearDout() ; // 出力をリセット

  UECSsetup();
  
  // W_RCopCond変数の異常でリセットの条件判断
  if((W_RCopCond<0)||(W_RCopCond>4)) {
    W_RCopCond = 4 ; //全値リセットに設定
    initializeSetpoints();
    }
  // 内部時刻のリセット
  innerTime = 0 ;
  secCount = 60 ;

  U_ccmList[CCMID_rcA_cRV].value = -1 ; // 遠隔制御CCMを無効化
  U_ccmList[CCMID_rcM_cRV].value = -1 ; // 遠隔操作CCMを無効化

  //天窓初期状態を全閉でスタート
  ventOpenPerOrg = 0.0 ;
  U_ccmList[CCMID_OpenP_cRV].value = (signed long)ventOpenPerOrg ;
  openOpsec = 0 ;
  closeOpsec = W_FcloseSec ; //　窓を全閉時間閉める指示
  
// ■■■■■ DEBUG
//Serial.begin(115200); // Baud Rate set to 115200
}

//------------------------------------------------------
// EPROM関係の制御設定値等をクリアする。
// W_modeの変数値がおかしい場合もリセット
//------------------------------------------------------
void initializeSetpoints() {
  W_RVcond = 0 ; // 換気窓開閉停止
  W_RVmode = 0 ; // 制御の停止
  W_RCopCond  = 0 ; // 強制動作とリセットも停止状態へ
  W_tempSP1 = 28 ; // 換気気温1
  W_timeSP2 = 2400 ;  // 開始時刻2
  W_tempSP2 = 28 ; // 換気気温2
  W_timeSP3 = 2400 ; // 開始時刻3
  W_tempSP3 = 28 ; // 換気気温3
  W_timeSP4 = 2400 ; // 開始時刻4
  W_tempSP4 = 28 ; // 換気気温4
  W_TempHis = 1 ; // 気温ヒステリシスを1℃
  W_FopenSec = 30 ;  // 全開所要時間
  W_FcloseSec = 28 ; // 全閉所要時間
  W_percent1 = 10 ;// 1回開閉量
  W_TDifAdj = 10 ; // 気温差感度(1開閉10%の最小値になる内外気温差)
}

//------------------------------------------------------
// 時刻の処理(1分毎に呼び出す)読めなかった時には内部時計を1分進める
//------------------------------------------------------
void checkTime1min() {
  if(!U_ccmList[CCMID_Time].validity) {
    // UECS-CCMが正しく受信できない場合、内部時計を1分増やす
    if((innerTime % 100) >= 59) { // 59分なら時間に1加える
      innerTime = (innerTime / 100 + 1) * 100 ;
      if(innerTime >= 2400) innerTime = 0 ; // 24時は0時に
    }
    else innerTime++ ; // それ以外は1分加える
  }
}
//------------------------------------------------------
// 時刻の読込(1秒毎に呼び出す)
//------------------------------------------------------
void checkTime1sec() {
  if(U_ccmList[CCMID_Time].validity) {
    innerTime = U_ccmList[CCMID_Time].value / 100 ; // 秒の単位を落として内部時計に代入
  }
}

//------------------------------------------------------
// 換気設定気温の計算(CCM変数に代入)
// 毎分呼出
//------------------------------------------------------
void calcVentSP() {
  int workSP ;
  workSP = W_tempSP1 ;
  if(innerTime >= W_timeSP2) workSP = W_tempSP2 ;
  if(innerTime >= W_timeSP3) workSP = W_tempSP3 ;
  if(innerTime >= W_timeSP4) workSP = W_tempSP4 ;
  U_ccmList[CCMID_VentSP_cRV].value = workSP ;
}

//------------------------------------------------------
// 降雨検出
// CCMのセット と　戻り値　降雨=1　毎秒チェック
//------------------------------------------------------
void rainCheck() {
  din() ;
  U_ccmList[CCMID_RainS_cRV].value = DIpin[rain] ;
}

//------------------------------------------------------
// 内外気温差から1回開閉量計算(感度計算)
// W_percent1に算出　毎分チェック
//------------------------------------------------------
void calcPercent1() {
  int workGA ;
  //内外気温差取得
  if((U_ccmList[CCMID_InAirTemp_mIC].validity)&&(U_ccmList[CCMID_WAirTemp_mOC].validity)) {
    // CCM取得値有効
    workGA = (U_ccmList[CCMID_InAirTemp_mIC].value - U_ccmList[CCMID_WAirTemp_mOC].value) / 10 ;
    if(workGA <= 1) W_percent1 = 100 ; // 換気冷却効果がない範囲は100%
    else if(workGA >= W_TDifAdj) W_percent1 = 10 ; // 感度調整範囲を超えたら10%
    else W_percent1 = 100 - (((workGA - 1) * 100) / (W_TDifAdj * 9) * 10) ; // 比例帯の感度計算
  }
  else W_percent1 = 20 ; // 内外気温差が取得できないときにはとりあえず20%に

// ■■■■■ DEBUG
//Serial.print("W_percent1=");Serial.print(W_percent1);Serial.print(", workGA=");Serial.print(workGA);
//Serial.print(", W_TDifAdj=");Serial.print(W_TDifAdj);Serial.print("\n");

}
  
//------------------------------------------------------
// 目標開度に必要な開閉時間を計算し
// opeOpsecまたはcloseOpsecにセットする
//------------------------------------------------------
void setOpSec(signed long aimPercent) {
  signed long workP ;
  workP = aimPercent /10 * 10 ; // 10%単位に切り捨て
  // 全開閉ならばズレを修正するため無条件に最大時間をセット
  if(workP >= 100) {
    openOpsec = W_FopenSec ;
    closeOpsec = 0 ;
    return ;
  }
  if(workP <= 0) {
    openOpsec = 0 ;
    closeOpsec = W_FcloseSec ;
    return ;
  }
  
  // 何%開閉すれば良いか10%単位で計算
  workP =  workP - U_ccmList[CCMID_OpenP_cRV].value / 10 * 10 ;
  
  if(workP == 0) { // 目標値なら制御停止
    openOpsec = closeOpsec = 0 ;
    return ;
  }
  if(workP > 0) { //開ける
    openOpsec = W_FopenSec * workP / 100 ; // 秒以下切り捨て
    closeOpsec = 0 ;
    return ;
  }
  if(workP < 0) { //閉める
    openOpsec = 0 ;
    closeOpsec = W_FcloseSec * workP / 100 * -1 ; // 秒以下切り捨て
    return ;
  }
}

//------------------------------------------------------
// 制御モードにより開閉時間をopenOpsecまたはcloseOpsecにセット
// 毎秒呼出
//------------------------------------------------------
void VentOpSet() {
  signed long workP ;
  
  // 停止の時はすべて止める
  if(W_RCopCond == 0) {
    openOpsec = closeOpsec = 0 ;
    W_RVmode = 0 ;
    return ;
  }
  // 強制開
  if(W_RCopCond == 2) {
    openOpsec = 1 ;
    closeOpsec = 0 ;
    W_RVmode = 2 ;
    return ;
  }
  // 強制閉
  if(W_RCopCond == 3) {
    openOpsec = 0 ;
    closeOpsec = 1 ;
    W_RVmode = 2 ;
    return ;
  }
  // 遠隔モードは開閉モータの停止1秒後から有効
  if((openOpsec < 0)&&(closeOpsec <  0)) {
    // 遠隔操作モード
    if((U_ccmList[CCMID_rcM_cRV].validity)&&(U_ccmList[CCMID_rcM_cRV].value <= 100)&&(U_ccmList[CCMID_rcM_cRV].value >= 0)) {
      setOpSec(U_ccmList[CCMID_rcM_cRV].value) ;
      W_RVmode = 4 ;
      return ;
    }
    // 遠隔制御モード
    if((U_ccmList[CCMID_rcA_cRV].validity)&&(U_ccmList[CCMID_rcA_cRV].value <= 100)&&(U_ccmList[CCMID_rcA_cRV].value >= 0)) {
      setOpSec(U_ccmList[CCMID_rcA_cRV].value) ;
      W_RVmode = 3 ;
      return ;
    }
  }
  // 自動モード
  if(W_RCopCond == 1) {  
    // 雨で強制閉
    if(U_ccmList[CCMID_RainS_cRV].value == 1) {
      // 窓が開いていれば閉める
      if(U_ccmList[CCMID_OpenP_cRV].value > 0) setOpSec(0) ;
      return ;
    }
    // 自動モードは開閉モータの停止1分後から有効
    if((openOpsec < -60)&&(closeOpsec <  -60)) {
      // 内外気温差のデータが有効で、室内気温が設定気温より高く、室外気温が室内気温以下ならば、
      if((U_ccmList[CCMID_InAirTemp_mIC].validity)&&(U_ccmList[CCMID_WAirTemp_mOC].validity)) {
        if(U_ccmList[CCMID_InAirTemp_mIC].value - ((U_ccmList[CCMID_VentSP_cRV].value + W_TempHis) * 10) > 0) {
          if(U_ccmList[CCMID_InAirTemp_mIC].value >= U_ccmList[CCMID_WAirTemp_mOC].value) {
            if(U_ccmList[CCMID_OpenP_cRV].value < 100) { // まだ全開ではない
              workP = U_ccmList[CCMID_OpenP_cRV].value + W_percent1 ;
              if(workP > 100) workP = 100 ; // 100%を超える時は100%に
              setOpSec(workP) ; // 窓を開ける
            }
          }
        }
        // 室内気温が設定気温より低く、室外気温が室内気温以下ならば、
        if(U_ccmList[CCMID_InAirTemp_mIC].value - ((U_ccmList[CCMID_VentSP_cRV].value - W_TempHis) * 10) < 0) {
          if(U_ccmList[CCMID_InAirTemp_mIC].value >= U_ccmList[CCMID_WAirTemp_mOC].value) {
            if(U_ccmList[CCMID_OpenP_cRV].value > 0) { // まだ全閉ではない
              workP = U_ccmList[CCMID_OpenP_cRV].value - W_percent1 ;
              if(workP < 0) workP = 0 ; // 0%を下回る時は0%に
              setOpSec(workP) ; // 窓を閉める
            }
          }
        }
      }
    }
    W_RVmode = 1 ;    
  }
}

//------------------------------------------------------
// openOpsecまたはcloseOpsecに基づいて制御の実施
// 毎秒呼出
//------------------------------------------------------
void VentControl() {  
  // 同時オンを防ぐ安全処理
  if((openOpsec > 0)&&(closeOpsec > 0)) closeOpsec = 0 ;
  // モーターオンオフの設定
  if(openOpsec > 0) {
    openMotor = HIGH ;
    W_RVcond = 1 ;
    // 精度落ちを防ぐため浮動小数点で計算
    ventOpenPerOrg = ventOpenPerOrg + (100.0 / (float)W_FopenSec) ;
    if(ventOpenPerOrg > 100.0) ventOpenPerOrg = 100.0 ;
  }
  else {
    openMotor = LOW ;
    W_RVcond = 0 ;
  }
  if(closeOpsec > 0) {
    closeMotor = HIGH ;
    W_RVcond = 2 ;    
    // 精度落ちを防ぐため浮動小数点で計算
    ventOpenPerOrg = ventOpenPerOrg - (100.0 / (float)W_FcloseSec) ;
    if(ventOpenPerOrg < 0.0) ventOpenPerOrg = 0.0 ;
  }
  else {
    closeMotor = LOW ;
    W_RVcond = 0 ;
  }
  // 計算した開度を代入
  U_ccmList[CCMID_OpenP_cRV].value = (signed long)(ventOpenPerOrg + 0.5) ;
  // 制御出力
  controlOut() ;
  // 値の減算
  openOpsec-- ;
  if(openOpsec < -3600) openOpsec = -3600 ;
  closeOpsec-- ;
  if(closeOpsec < -3600) closeOpsec = -3600 ;
}

//---------------------------------------------------------
//Webページから入力が行われ各種値を設定し不揮発性メモリに値を保存後、
//ページの再描画を行う前に以下の関数が呼び出される。
//---------------------------------------------------------
void OnWebFormRecieved(){
  
  // 初期化の処理
  if(W_RCopCond == 4) {
    initializeSetpoints();
    // 内部時刻のリセット
    innerTime = 0 ;
    secCount = 60 ;
    
    U_ccmList[CCMID_rcA_cRV].value = -1 ; // 遠隔制御CCMを無効化
    U_ccmList[CCMID_rcM_cRV].value = -1 ; // 遠隔操作CCMを無効化

    //天窓初期状態を全閉でスタート
    ventOpenPerOrg = 0.0 ;
    U_ccmList[CCMID_OpenP_cRV].value = (signed long)ventOpenPerOrg ;
    openOpsec = 0 ;
    closeOpsec = W_FcloseSec ; //　窓を全閉時間閉める指示
  }
  
  // 気温差感度を変更した場合のため、再計算
  calcPercent1() ;
}

//------------------------------------------------------
// 制御で代入していないCCMとリセット後の秒数Esecをセット
// 毎秒チェック
//------------------------------------------------------
void setCCM(){
  // 状態に窓開閉状態をセット
  U_ccmList[CCMID_cnd_cRV].value = W_RVcond ;
  // WDT動作確認用
  U_ccmList[CCMID_esec_cRV].value = (signed long)(millis() / 1000) ;
}

//---------------------------------------------------------
//１秒に１回呼び出される関数
//---------------------------------------------------------
void UserEverySecond(){

  //ウオッチドッグタイマークリア
  clearWdt() ;
  checkTime1sec() ; // 時間をUPDATE
  
  if(secCount >= 60) {
    //////////////////////////////////////////////////////
    // 1分毎の処理
    // void UserEveryMinute()は誤差がある可能性があるので
    // 使用しない(1回給液しなかったりすると大変!)
    //////////////////////////////////////////////////////
    secCount = 0 ; // 秒カウンターをクリア
    checkTime1min() ;
        
    // 日界処理　= 0:00の処理
    if(innerTime == 0) {
    }
    
    calcVentSP() ; // 換気気温設定値計算
    calcPercent1() ; // 気温差感度調節
  }
  
  /////////////////////
  // 以下は1秒毎の処理
  /////////////////////
  rainCheck() ; // 感雨
  VentOpSet() ; // 窓制御算出処理メイン

// ■■■■■ DEBUG
//Serial.print("openOpsec=");Serial.print(openOpsec);Serial.print(", closeOpsec=");Serial.print(closeOpsec);Serial.print("\n");

  VentControl() ; // 制御実行
  setCCM() ; // CCMに値セット
  secCount++ ; // 秒カウンターを1up
  
}

//---------------------------------------------------------
//１分に１回呼び出される関数
//---------------------------------------------------------
void UserEveryMinute(){
}

//---------------------------------------------------------
//メインループ
//システムのタイマカウント，httpサーバーの処理，
//UDP16520番ポートと16529番ポートの通信文をチェックした後，呼び出さされる関数。
//呼び出される頻度が高いため，重い処理を記述しないこと。
//---------------------------------------------------------
void UserEveryLoop(){
}

//---------------------------------------------------------
//setup()実行後に呼び出されるメインループ
//この関数内ではUECSloop()関数を呼び出さなくてはならない。
//UserEveryLoop()に似ているがネットワーク関係の処理を行う前に呼び出される。
//必要に応じて処理を記述してもかまわない。
//呼び出される頻度が高いため,重い処理を記述しないこと。
//---------------------------------------------------------
void loop(){
  UECSloop();
}

