/*
 Low-cost UECS CO2 fertilizer and heater control node
 LU-CO2HEAT By T.Hoshi, Kindai Univ.
 2017.11.08 version 1.0
 2017.11.13 yesterday operating time added version 1.1
*/

// このプログラムは、無償提供する代わりに、動作結果に関する
// 一切の保証は致しません。自己責任でお使いください。また、
// 改造も自由ですが、公刊等をされる場合は、「近畿大学　星　
// 岳彦が開発した○○〇プログラムを用いた」と引用して下さい
// ますと、今後の研究開発の励みになりますので、どうぞよろ
// しくお願いします。　　　　　2019.3.4 Takehiko Hoshi
// 
// このシールドはArduino MEGA 2560 + Ethernet Shiedld2の
// 組み合わせで動作させることを前提にしています。
// プログラムを書き込むときには、Arduino MEGA 2560だけ
// の状態で書き込んでください。
// Ethernet Sheild 2基板のシールのmacアドレスに変更して
// ください。このプログラムをmacで検索すると付近にあり
// ます。macアドレスとは。90:A2:DA:0F:xx:xx のような8個
// の16進数です。これをばらして0xの後に転記してください。

// 出力の割り付け状態
// Dout0 液化CO2供給　電磁弁ON信号
// Dout1 なし
// Dout2 温風暖房機　運転(暖房)信号
// Dout3 温風暖房機ファン　動作信号
// Din3 暖房機警報入力　警報で電圧入力

#include <SPI.h>
#include <Ethernet2.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <EEPROM.h>
// UECS用のライブラリUARDECSのインストール
// https://uecs.org/arduino/uardecs.html が必要です。
#include <Uardecs.h>

//////////////////////
// 定数関連定義
//////////////////////

//制御出力の定義
const uint8_t CO2out = 0 ; // CO2電磁弁
const uint8_t fanout = 3 ; // 暖房機循環FAN
const uint8_t heaterout = 2 ; // 暖房機運転
const uint8_t heaterErr = 3 ; // 暖房機警報入力

//
//IPアドレスをリセットするピンの割付
//
//Pin ID. This pin is pull-upped automatically.
const byte U_InitPin = 57 ; // for Arduino MEGA 2560 
//const byte U_InitPin = 17 ; // for Arduino Ethernet
//This shields is assigned as A3 for reset pin.
//When U_InitPin status equals this value,IP address is set "192.168.1.7".
//(This is added Ver3 over)
const byte U_InitPin_Sense= HIGH ;  

/////////////////////////////
// ノードの基礎情報の設定
/////////////////////////////

const char U_name[] PROGMEM =   "LU-CO2HEAT";
const char U_vender[] PROGMEM = "Hoshi-lab.";
const char U_uecsid[] PROGMEM = "011009001003";
const char U_footnote[] PROGMEM = "UARDECS7.5 Product by <A HREF=\"https://hoshi-lab.info/\">Kindai, hoshi-lab.</A>";
char U_nodename[20] = "LU-CO2HEAT";
UECSOriginalAttribute U_orgAttribute;

////////////////////////////////////////////
// CO2HEATノードの変数設定とライブラリ
////////////////////////////////////////////

// 入出力ポートの定義
const int port[4] = { 7, 8, 5, 6 } ;
// ウオッチドッグタイマーのリセットポート
const int wdt = 9 ; // 0 = clear timer
// ディジタル出力値の一時記憶装置のトリガポート
const int ratch = 3 ; //  0->1 ratch
// ディジタル入力値のイネーブルポート
const int in_out = 2 ; // in = 1 out =0

//制御関係定数・変数
const int C_CO2presec = 30 ; // CO2の注入開始までの待ち時間(秒)
int CO2workTime ; // 施用していないとき0, 施用開始からの秒数
                  // この値で送風ファンとCO2電磁弁の制御を実施
int CO2_supply, FAN_operate, HEATER_operate ; // 各運転状況フラグ HIGHで運転
int innerTime ; // 内部時刻hhmm
int secCount ; // 60秒に一回の処理

// 作業変数
int i ; // ループ用変数
int DOpin[4] ; // 制御出力配列
int DIpin[4] ; // ディジタル入力配列

//Define variables for Web interface
signed long W_heatcond ; // 暖房制御状態
signed long W_DheatTime ; // 昨日暖房時間
signed long W_CO2cond ; // CO2制御状態
signed long W_DCO2supTime ; // 昨日CO2供給時間
signed long W_Chmode ; // 自動モード
signed long W_tempSP1 ; // 暖房気温1
signed long W_timeSP2 ; // 開始時刻2
signed long W_tempSP2 ; // 暖房気温2
signed long W_timeSP3 ; // 開始時刻3
signed long W_tempSP3 ; // 暖房気温3
signed long W_timeSP4 ; // 開始時刻4
signed long W_tempSP4 ; // 暖房気温4
signed long W_TempHis ; // 気温幅±
signed long W_CO2injsec ; // CO2注入時間
signed long W_CO2postsec ; // CO2後待時間
signed long W_CO2tempOpen ; // CO2上限気温
signed long W_CO2tempPhsy ; // CO2下限気温
signed long W_CO2solarPhsy ; // CO2下限日射
signed long W_CO2highSP ; // 設定CO2濃度

// ウオッチドッグタイマーのクリア(2秒以内に1回の実行が必要)
void clearWdt() {
  digitalWrite(wdt, LOW) ;
  delayMicroseconds(100) ;
  digitalWrite(wdt, HIGH) ;
}

// ディジタル値を出力する d[0]-d[3]の値 0はOFF、1はON
void dout(int *d) {
  digitalWrite(in_out, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], OUTPUT) ;
    digitalWrite(port[i], d[i]) ;
  }
  delayMicroseconds(1) ;
  digitalWrite(ratch, HIGH) ;
  delayMicroseconds(1) ; 
  digitalWrite(ratch, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], INPUT) ;
  }
  digitalWrite(in_out, HIGH) ;
}

// 制御出力を全てOFFにする
void clearDout() {
  int i;
  for(i=0; i<4; i++) DOpin[i] = LOW ;
  dout(DOpin) ;
  CO2_supply = FAN_operate = HEATER_operate = LOW ;
}

// フラグに従い出力
void controlOut() {
  DOpin[CO2out] = CO2_supply ;
  DOpin[fanout] = FAN_operate ;
  DOpin[heaterout] = HEATER_operate ;
  dout(DOpin) ;
}
 
// ディジタル入力値の読み込み d[0]-d[3]に値を代入
void din() {
  digitalWrite(in_out, HIGH) ;
  delayMicroseconds(1) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], INPUT) ;
    delayMicroseconds(1) ;
    DIpin[i] = digitalRead(port[i]) ;
  }
}

//////////////////////////////////
// html page setting
//////////////////////////////////

// Define Strings
const char S_heatcond[] PROGMEM ="暖房状態" ;
const char S_DheatTime[] PROGMEM ="昨日暖房" ;
const char S_CO2cond[] PROGMEM ="CO2状態" ;
const char S_DCO2supTime[] PROGMEM ="昨日CO2" ;
const char S_Chmode[] PROGMEM ="自動モード" ;
const char S_tempSP1[] PROGMEM ="暖房気温1" ;
const char S_timeSP2[] PROGMEM ="開始時刻2" ;
const char S_tempSP2[] PROGMEM ="暖房気温2" ;
const char S_timeSP3[] PROGMEM ="開始時刻3" ;
const char S_tempSP3[] PROGMEM ="暖房気温3" ;
const char S_timeSP4[] PROGMEM ="開始時刻4" ;
const char S_tempSP4[] PROGMEM ="暖房気温4" ;
const char S_TempHis[] PROGMEM ="気温幅±" ;
const char S_CO2injsec[] PROGMEM ="CO2注入" ;
const char S_CO2postsec[] PROGMEM ="CO2後待" ;
const char S_CO2tempOpen[] PROGMEM ="上限気温" ;
const char S_CO2tempPhsy[] PROGMEM ="下限気温" ;
const char S_CO2solarPhsy[] PROGMEM ="下限日射" ;
const char S_CO2highSP[] PROGMEM ="CO2濃度" ;

//Select Strings
const char S_off[] PROGMEM = "停止";
const char S_fan[] PROGMEM = "送風";
const char S_heat[] PROGMEM = "暖房";
const char S_preCO2[] PROGMEM = "前送風";
const char S_inCO2[] PROGMEM = "供給";
const char S_postCO2[] PROGMEM = "後送風";
const char S_autoheat[] PROGMEM = "暖房";
const char S_autoCO2[] PROGMEM = "CO2";
const char S_autoCO2Heat[] PROGMEM = "暖房+CO2";
const char S_forcefan[] PROGMEM = "強制送風";
const char S_forceheat[] PROGMEM = "強制暖房";
const char S_forceCO2[] PROGMEM = "強制CO2";
const char S_reset[] PROGMEM = "初期化";
const char *SS_heatcond[3]={ S_off, S_fan, S_heat };
const char *SS_CO2cond[4]={ S_off, S_preCO2, S_inCO2, S_postCO2 };
const char *SS_Chmode[8]={ S_off, S_autoheat, S_autoCO2, S_autoCO2Heat, S_forcefan, S_forceheat, S_forceCO2, S_reset };

//Unit Strings
const char U_sec[] PROGMEM =  "秒";
const char U_temp[] PROGMEM =  "℃";
const char U_time[] PROGMEM =  "時.分";
const char U_flux[] PROGMEM =  "kW/m2";
const char U_ppm[] PROGMEM =  "ppm";

//Detail Strings
const char D_init[] PROGMEM =   "初期値";
const char D_fb2400[] PROGMEM =   "24.00無視";
const char D_hist[] PROGMEM =   "不感帯";
const char D_w30[] PROGMEM =   "30秒後に";
const char D_wmix[] PROGMEM =   "混合待ち";
const char D_upLow[] PROGMEM =   "以上低濃度";
const char D_downLow[] PROGMEM =   "以下低濃度";
const char D_high[] PROGMEM =   "高濃度時";

// Dummy Strings
const char EMPTY[] PROGMEM= "";
const char** DUMMY = NULL;

const int U_HtmlLine = 19; //Total number of HTML table rows.

struct UECSUserHtml U_html[U_HtmlLine] = {
//{  名前,          入出力形式,     単位,  詳細説明,   選択肢文字列,選択肢数, 値,               最小値,最大値, 小数桁数}
  { S_heatcond,     UECSSHOWSTRING, EMPTY,  EMPTY,     SS_heatcond, 3,        &(W_heatcond),    0,     2,      0},
  { S_DheatTime,    UECSSHOWDATA,   U_sec,  EMPTY,     DUMMY,       0,        &(W_DheatTime),   0,     0,      0},
  { S_CO2cond,      UECSSHOWSTRING, EMPTY,  EMPTY,     SS_CO2cond,  4,        &(W_CO2cond),     0,     3,      0},
  { S_DCO2supTime,  UECSSHOWDATA,   U_sec,  EMPTY,     DUMMY,       0,        &(W_DCO2supTime), 0,     0,      0},
  { S_Chmode,       UECSSELECTDATA, EMPTY,  EMPTY,     SS_Chmode,   8,        &(W_Chmode),      0,     7,      3},
  { S_tempSP1,      UECSINPUTDATA,  U_temp, D_init,    DUMMY,       0,        &(W_tempSP1),     0,     30,     0},
  { S_timeSP2,      UECSINPUTDATA,  U_time, D_fb2400,  DUMMY,       0,        &(W_timeSP2),     0,     2400,   2},
  { S_tempSP2,      UECSINPUTDATA,  U_temp, EMPTY,     DUMMY,       0,        &(W_tempSP2),     0,     30,     0},
  { S_timeSP3,      UECSINPUTDATA,  U_time, D_fb2400,  DUMMY,       0,        &(W_timeSP3),     0,     2400,   2},
  { S_tempSP3,      UECSINPUTDATA,  U_temp, EMPTY,     DUMMY,       0,        &(W_tempSP3),     0,     30,     0},
  { S_timeSP4,      UECSINPUTDATA,  U_time, D_fb2400,  DUMMY,       0,        &(W_timeSP4),     0,     2400,   2},
  { S_tempSP4,      UECSINPUTDATA,  U_temp, EMPTY,     DUMMY,       0,        &(W_tempSP4),     0,     30,     0},
  { S_TempHis,      UECSINPUTDATA,  U_temp, D_hist,    DUMMY,       0,        &(W_TempHis),     0,     10,     0},
  { S_CO2injsec,    UECSINPUTDATA,  U_sec,  D_w30,     DUMMY,       0,        &(W_CO2injsec),   0,     100,    0},
  { S_CO2postsec,   UECSINPUTDATA,  U_sec,  D_wmix,    DUMMY,       0,        &(W_CO2postsec),  0,     1800,   0},
  { S_CO2tempOpen,  UECSINPUTDATA,  U_temp, D_upLow,   DUMMY,       0,        &(W_CO2tempOpen), 0,     40,     0},  
  { S_CO2tempPhsy,  UECSINPUTDATA,  U_temp, D_downLow, DUMMY,       0,        &(W_CO2tempPhsy), 0,     40,     0},
  { S_CO2solarPhsy, UECSINPUTDATA,  U_flux, D_downLow, DUMMY,       0,        &(W_CO2solarPhsy),0,     2000,   3},
  { S_CO2highSP,    UECSINPUTDATA,  U_ppm,  D_high,    DUMMY,       0,        &(W_CO2highSP),   0,     2000,   0},
} ;

//////////////////////////////////
// UserCCM setting
//////////////////////////////////

//define CCMID for identify
//CCMID_dummy must put on last
enum { 
  CCMID_InAirTemp_mIC, CCMID_InAirCO2_mIC,   CCMID_WRadiation_mOC,
  CCMID_Time,          CCMID_cnd_cCH,        CCMID_HeatTime_cCH,
  CCMID_HeatSP_cCH,    CCMID_CO2supTime_cCH, CCMID_CO2SP_cCH,
  CCMID_esec_cCH,      CCMID_dummy
};
const int U_MAX_CCM = CCMID_dummy;
UECSCCM U_ccmList[U_MAX_CCM];

const char ccmNameInAirTemp[] PROGMEM = "室内気温";
const char ccmTypeInAirTemp[] PROGMEM = "InAirTemp.mIC";
const char ccmUnitInAirTemp[] PROGMEM = "C";

const char ccmNameInAirCO2[] PROGMEM = "CO2濃度";
const char ccmTypeInAirCO2[] PROGMEM = "InAirCO2.mIC";
const char ccmUnitInAirCO2[] PROGMEM = "ppm";

const char ccmNameWRadiation[] PROGMEM = "屋外日射";
const char ccmTypeWRadiation[] PROGMEM = "WRadiation.mOC";
const char ccmUnitWRadiation[] PROGMEM = "kW/m2";

const char ccmNameTime[] PROGMEM = "時刻";
const char ccmTypeTime[] PROGMEM = "Time";
const char ccmUnitTime[] PROGMEM = "hhmmss";

const char ccmNamecnd[] PROGMEM = "状態";
const char ccmTypecnd[] PROGMEM = "cnd.cCH";
const char ccmUnitcnd[] PROGMEM = "";

const char ccmNameHeatTime[] PROGMEM = "暖房時間";
const char ccmTypeHeatTime[] PROGMEM = "HeatTime.cCH";
const char ccmUnitHeatTime[] PROGMEM = "s/d-1";

const char ccmNameHeatSP[] PROGMEM = "暖房気温";
const char ccmTypeHeatSP[] PROGMEM = "HeatSP.cCH";
const char ccmUnitHeatSP[] PROGMEM = "C";

const char ccmNameCO2supTime[] PROGMEM = "CO2供給時間";
const char ccmTypeCO2supTime[] PROGMEM = "CO2supTime.cCH";
const char ccmUnitCO2supTime[] PROGMEM = "s/d-1";

const char ccmNameCO2SP[] PROGMEM = "CO2供給濃度";
const char ccmTypeCO2SP[] PROGMEM = "CO2SP.cCH";
const char ccmUnitCO2SP[] PROGMEM = "ppm";

// WDT動作確認用
const char ccmNameEsec[] PROGMEM = "経過時間";
const char ccmTypeEsec[] PROGMEM = "esec.cCH";
const char ccmUnitsec[] PROGMEM = "s";

//------------------------------------------------------
//UARDECS初期化用関数
//主にCCMの作成とMACアドレスの設定を行う
//------------------------------------------------------
void UserInit(){

 //Set ccm list
 //         send,  CCMID,                Name,              Type Name,         Unit,              Pri, dig, class
 UECSsetCCM(false, CCMID_InAirTemp_mIC,  ccmNameInAirTemp,  ccmTypeInAirTemp,  ccmUnitInAirTemp,  30,  1,   A_10S_0 );
 UECSsetCCM(false, CCMID_InAirCO2_mIC,   ccmNameInAirCO2,   ccmTypeInAirCO2,   ccmUnitInAirCO2,   30,  0,   A_10S_0 );
 UECSsetCCM(false, CCMID_WRadiation_mOC, ccmNameWRadiation, ccmTypeWRadiation, ccmUnitWRadiation, 30,  3,   A_10S_0 );
 UECSsetCCM(false, CCMID_Time,           ccmNameTime,       ccmTypeTime,       ccmUnitTime,       30,  0,   A_1M_0  );
 UECSsetCCM(true,  CCMID_cnd_cCH,        ccmNamecnd,        ccmTypecnd,        ccmUnitcnd,        30,  0,   A_1S_0  );
 UECSsetCCM(true,  CCMID_HeatTime_cCH,   ccmNameHeatTime,   ccmTypeHeatTime,   ccmUnitHeatTime,   30,  0,   A_10S_0 );
 UECSsetCCM(true,  CCMID_HeatSP_cCH,     ccmNameHeatSP,     ccmTypeHeatSP,     ccmUnitHeatSP,     30,  0,   A_1M_0  );
 UECSsetCCM(true,  CCMID_CO2supTime_cCH, ccmNameCO2supTime, ccmTypeCO2supTime, ccmUnitCO2supTime, 30,  0,   A_10S_0 );
 UECSsetCCM(true,  CCMID_CO2SP_cCH,      ccmNameCO2SP,      ccmTypeCO2SP,      ccmUnitCO2SP,      30,  0,   A_10S_0 );
 UECSsetCCM(true,  CCMID_esec_cCH,       ccmNameEsec,       ccmTypeEsec,       ccmUnitsec,        30,  0,   A_1S_0  );  // WDT動作確認用

 //Set proper MAC address
 U_orgAttribute.mac[0] = 0x90;
 U_orgAttribute.mac[1] = 0xA2;
 U_orgAttribute.mac[2] = 0xDA;
 U_orgAttribute.mac[3] = 0x11;
 U_orgAttribute.mac[4] = 0x1D;
 U_orgAttribute.mac[5] = 0x0A;

}

//---------------------------------------------------------
//起動直後に１回呼び出される関数。
//様々な初期化処理を記述できる。
//この関数内ではUECSsetup()関数を呼び出さなくてはならない。
//必要に応じて処理を記述してもかまわない。
//---------------------------------------------------------
void setup(){

  //制御シールドの制御信号線の初期化
  pinMode(in_out, OUTPUT) ;
  digitalWrite(in_out, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], OUTPUT) ;
    digitalWrite(port[i], LOW) ;
  }
  pinMode(ratch, OUTPUT) ;
  digitalWrite(ratch, HIGH) ;
  delayMicroseconds(1) ; 
  digitalWrite(ratch, LOW) ;
  pinMode(wdt, OUTPUT) ;
  
  digitalWrite(wdt, HIGH) ; // ウオッチドッグクリア信号を初期値にする
  clearWdt() ;
  clearDout() ; // 出力をリセット

  UECSsetup();
  
  // W_mode変数の異常でリセットの条件判断
  if((W_Chmode<0)||(W_Chmode>7)) {
    W_Chmode = 7 ; //全値リセットに設定
    initializeSetpoints();
    }
  // 内部時刻のリセット
  innerTime = 0 ;
  secCount = 60 ;
  // CO2作業変数をリセット
  CO2workTime = 0 ;
}

//------------------------------------------------------
// EPROM関係の制御設定値等をクリアする。
// W_modeの変数値がおかしい場合もリセット
//------------------------------------------------------
void initializeSetpoints() {
  W_heatcond = 0 ; // 暖房制御状態
  W_DheatTime = 0 ; // 昨日暖房
  W_CO2cond  = 0 ; // CO2制御状態
  W_DCO2supTime = 0 ; // 昨日CO2供給
  W_Chmode = 0 ; // 自動モード
  W_tempSP1 = 8 ; // 暖房気温1
  W_timeSP2 = 2400 ;  // 開始時刻2
  W_tempSP2 = 8 ; // 暖房気温2
  W_timeSP3 = 2400 ; // 開始時刻3
  W_tempSP3 = 8 ; // 暖房気温3
  W_timeSP4 = 2400 ; // 開始時刻4
  W_tempSP4 = 8 ; // 暖房気温4
  W_TempHis = 1 ; // 気温幅±
  W_CO2injsec = 10 ;  // CO2注入時間
  W_CO2postsec= 300 ; // CO2後待時間
  W_CO2tempOpen = 28 ;// CO2上限気温
  W_CO2tempPhsy = 10 ; // CO2下限気温
  W_CO2solarPhsy = 100 ; // CO2下限日射
  W_CO2highSP = 1000 ; // 設定CO2濃度

  U_ccmList[CCMID_HeatTime_cCH].value = 0 ; // 暖房時間クリア
  U_ccmList[CCMID_CO2supTime_cCH].value = 0 ; // CO2供給時間クリア
}

//------------------------------------------------------
// 時刻の処理(1分毎に呼び出す)読めなかった時には内部時計を1分進める
//------------------------------------------------------
void checkTime1min() {
  if(!U_ccmList[CCMID_Time].validity) {
    // UECS-CCMが正しく受信できない場合、内部時計を1分増やす
    if((innerTime % 100) >= 59) { // 59分なら時間に1加える
      innerTime = (innerTime / 100 + 1) * 100 ;
      if(innerTime >= 2400) innerTime = 0 ; // 24時は0時に
    }
    else innerTime++ ; // それ以外は1分加える
  }
}
//------------------------------------------------------
// 時刻の読込(1秒毎に呼び出す)
//------------------------------------------------------
void checkTime1sec() {
  if(U_ccmList[CCMID_Time].validity) {
    innerTime = U_ccmList[CCMID_Time].value / 100 ; // 秒の単位を落として内部時計に代入
  }
}

//------------------------------------------------------
// 暖房設定気温の計算(CCM変数に代入)
// 毎分呼出
//------------------------------------------------------
void calcHeatSP() {
  int workSP ;
  workSP = W_tempSP1 ;
  if(innerTime >= W_timeSP2) workSP = W_tempSP2 ;
  if(innerTime >= W_timeSP3) workSP = W_tempSP3 ;
  if(innerTime >= W_timeSP4) workSP = W_tempSP4 ;
  U_ccmList[CCMID_HeatSP_cCH].value = workSP ;
}

//------------------------------------------------------
// CO2供給設定濃度の計算(CCM変数に代入)
// 毎分呼出
//------------------------------------------------------
void calcCO2SP() {
  int workSP ;
  if(CO2workTime == 0) { // CO2供給関係作業中でないときだけ計算
    workSP = 400 ; // 何も指定がないときは400ppmに(低濃度)
    //気温条件
    if(U_ccmList[CCMID_InAirTemp_mIC].validity) {
      if(U_ccmList[CCMID_InAirTemp_mIC].value < (W_CO2tempOpen * 10)) {
        if(U_ccmList[CCMID_InAirTemp_mIC].value > (W_CO2tempPhsy * 10)) {
          //日射条件
          if(U_ccmList[CCMID_WRadiation_mOC].validity) {
            if(U_ccmList[CCMID_WRadiation_mOC].value > W_CO2solarPhsy) {
              workSP = W_CO2highSP ; // 全ての条件成立で高濃度
            }
          }
        }
      }
    }
    U_ccmList[CCMID_CO2SP_cCH].value = workSP ;
  }
}
  
//------------------------------------------------------
// 状態とリセット後の秒数EsecをCCM送出のためセット
// 毎秒チェック
//------------------------------------------------------
void setCCM(){
  // 制御稼働時間設定
  if(HEATER_operate == HIGH)  U_ccmList[CCMID_HeatTime_cCH].value += 1 ; // 暖房
  if(CO2_supply == HIGH)  U_ccmList[CCMID_CO2supTime_cCH].value += 1 ; // CO2

  // cndに制御モード値をセット
  U_ccmList[CCMID_cnd_cCH].value = W_Chmode ;
  // 暖房機エラーなら強制的に999にする
  din() ;
  if(DIpin[heaterErr] == HIGH) U_ccmList[CCMID_cnd_cCH].value = 999 ;
  
  U_ccmList[CCMID_esec_cCH].value = (signed long)(millis() / 1000) ; // WDT動作確認用
}

//------------------------------------------------------
// 暖房自動制御
// 毎分呼出
//------------------------------------------------------
void heatauto() {
  // 自動暖房モードのときだけ実行
  if((W_Chmode == 1)||(W_Chmode == 3)) {     
    // 室内気温CCMが有効の時はヒステリシスを考えて暖房のオンオフ
    if(U_ccmList[CCMID_InAirTemp_mIC].validity) {
      // 室内気温CCMが有効の時はヒステリシスを考えて暖房のオンオフ
      if((U_ccmList[CCMID_InAirTemp_mIC].value / 10) < (U_ccmList[CCMID_HeatSP_cCH].value - W_TempHis)) {
        HEATER_operate = HIGH ;
        W_heatcond = 2 ; // 暖房制御状態を暖房に
      }
      if((U_ccmList[CCMID_InAirTemp_mIC].value / 10) > (U_ccmList[CCMID_HeatSP_cCH].value + W_TempHis)) {
        HEATER_operate = LOW ;
        W_heatcond = 0 ; // 暖房制御状態を停止に
      }
    }
  }
}

//------------------------------------------------------
// CO2自動制御
// 毎秒呼出
//------------------------------------------------------
void CO2auto() {
  // 自動か強制のCO2供給モードのときだけ実行
  if((W_Chmode == 2)||(W_Chmode == 3)||(W_Chmode == 6)) {
     if(CO2workTime > 0) { // CO2供給動作中処理
       FAN_operate = HIGH ;
       W_CO2cond = 1 ; // 前送風
       if((CO2workTime > C_CO2presec)&&(CO2workTime <= (C_CO2presec + W_CO2injsec))) {
         // CO2注入タイミングか?
         CO2_supply = HIGH ; // CO2注入時間帯だけON
         W_CO2cond = 2 ; // 供給
       }
       else {
         CO2_supply = LOW ; // 注入停止
       }
       if(CO2workTime > (C_CO2presec + W_CO2injsec)) { // 拡散待ちか
         W_CO2cond = 3 ; // 後送風
       }
       if(CO2workTime > (C_CO2presec + W_CO2injsec + W_CO2postsec)) { // 供給サイクル完了
         CO2workTime = 0 ;
         FAN_operate = LOW ;
         W_CO2cond = 0 ; // 停止
       }
       else CO2workTime++ ; // 完了でなければ1秒進める
     }
     else { // CO2を供給していないときには濃度チェック
       if(W_Chmode == 6) { // 強制動作モードのままならば、供給サイクルを再起動
           FAN_operate = HIGH ;
           CO2workTime = 1 ;
           W_CO2cond = 1 ; // 前送風
       }
       else {
         if(U_ccmList[CCMID_InAirCO2_mIC].validity) {  // CCMが有効で設定濃度を下回っていれば供給
           if(U_ccmList[CCMID_InAirCO2_mIC].value < U_ccmList[CCMID_CO2SP_cCH].value) {
             FAN_operate = HIGH ;
             CO2workTime = 1 ;
             W_CO2cond = 1 ; // 前送風
           }
         }
       }
     }
  }
  // 暖房機状態との連係動作
  // 暖房しておらず、ファン出力がオンならば、暖房制御状態を送風にする
  if((HEATER_operate == LOW)&&(FAN_operate == HIGH)) W_heatcond = 1 ;
  // 暖房しておらず、ファン出力もオフならば、暖房制御状態を停止にする  
  if((HEATER_operate == LOW)&&(FAN_operate == LOW)) W_heatcond = 0 ;
}
         
//---------------------------------------------------------
//Webページから入力が行われ各種値を設定し不揮発性メモリに値を保存後、
//ページの再描画を行う前に以下の関数が呼び出される。
//---------------------------------------------------------
void OnWebFormRecieved(){
  
  // リセットする場合の処理
  if(W_Chmode == 7) {
    initializeSetpoints() ;
    // 内部時刻のリセット
    innerTime = 0 ;
    secCount = 60 ;
    // CO2作業変数をリセット
    CO2workTime = 0 ;
  }
  
  // 強制停止処理
  if(W_Chmode == 0) {
    clearDout() ; // 全ての制御出力OFF
    W_heatcond = 0 ; // 暖房制御状態を停止
    CO2workTime = 0 ; // CO2供給強制中止
    W_CO2cond = 0 ; // CO2供給状態を停止
  }
  // 暖房のみ自動に設定したときにCO2供給中だった時の停止処理
    if(W_Chmode == 1) {
    CO2_supply = FAN_operate = LOW ;
    CO2workTime = 0 ; // CO2供給強制中止
    W_CO2cond = 0 ; // CO2供給状態を停止
  }
  // CO2のみ自動に設定したときに暖房中だった時の停止処理
    if(W_Chmode == 2) {
    HEATER_operate = LOW ;
    W_heatcond = 0 ; // 暖房制御状態を停止
  }
  // 強制送風動作
  if(W_Chmode == 4) {
    CO2_supply = HEATER_operate = LOW ;
    FAN_operate = HIGH ;
    W_heatcond = 1 ; // 暖房制御状態を送風に
    CO2workTime = 0 ; // CO2供給強制中止
    W_CO2cond = 0 ; // CO2供給状態を停止
  }
  // 強制暖房動作
  if(W_Chmode == 5) {
    CO2_supply = FAN_operate = LOW ;
    HEATER_operate = HIGH ;
    W_heatcond = 2 ; // 暖房制御状態を暖房に
    CO2workTime = 0 ; // CO2供給強制中止
    W_CO2cond = 0 ; // CO2供給状態を停止
  }
  // 強制CO2供給動作
  if(W_Chmode == 6) {
    CO2_supply = HEATER_operate = LOW ;
    FAN_operate = HIGH ;
    W_heatcond = 1 ; // 暖房制御状態を送風に
    CO2workTime = 1 ; // CO2供給を開始に
    W_CO2cond = 1 ; // CO2供給状態を前送風に
  }

}

//---------------------------------------------------------
//１秒に１回呼び出される関数
//---------------------------------------------------------
void UserEverySecond(){

  //ウオッチドッグタイマークリア
  clearWdt() ;
  checkTime1sec() ; // 時間をUPDATE
  
  if(secCount >= 60) {
    //////////////////////////////////////////////////////
    // 1分毎の処理
    // void UserEveryMinute()は誤差がある可能性があるので
    // 使用しない(1回給液しなかったりすると大変!)
    //////////////////////////////////////////////////////
    secCount = 0 ; // 秒カウンターをクリア
    checkTime1min() ;
    
    
    // 日界処理　= 0:00の処理
    if(innerTime == 0) {
      W_DheatTime = U_ccmList[CCMID_HeatTime_cCH].value ; // 昨日暖房時間セット
      U_ccmList[CCMID_HeatTime_cCH].value = 0 ; // 日暖房時間クリア
      W_DCO2supTime = U_ccmList[CCMID_CO2supTime_cCH].value ; //昨日CO2供給時間セット
      U_ccmList[CCMID_CO2supTime_cCH].value = 0 ; // 日CO2供給時間クリア
    }
    calcHeatSP() ; // 暖房制御処理
    heatauto() ;
  }
  
  /////////////////////
  // 以下は1秒毎の処理
  /////////////////////
  calcCO2SP() ; // CO2設定値計算
  CO2auto() ; // CO2自動制御
  controlOut() ; // 制御信号出力
  setCCM() ; // CCMに値セット
  secCount++ ; // 秒カウンターを1up
  
}

//---------------------------------------------------------
//１分に１回呼び出される関数
//---------------------------------------------------------
void UserEveryMinute(){
}

//---------------------------------------------------------
//メインループ
//システムのタイマカウント，httpサーバーの処理，
//UDP16520番ポートと16529番ポートの通信文をチェックした後，呼び出さされる関数。
//呼び出される頻度が高いため，重い処理を記述しないこと。
//---------------------------------------------------------
void UserEveryLoop(){
}

//---------------------------------------------------------
//setup()実行後に呼び出されるメインループ
//この関数内ではUECSloop()関数を呼び出さなくてはならない。
//UserEveryLoop()に似ているがネットワーク関係の処理を行う前に呼び出される。
//必要に応じて処理を記述してもかまわない。
//呼び出される頻度が高いため,重い処理を記述しないこと。
//---------------------------------------------------------
void loop(){
  UECSloop();
}

