/*
 Low-cost UECS curtain controller version 2
 LU-CURCON-2 By T.Hoshi, Kindai Univ.
 2015.12.01 version 1.0
 2016.02.09 version 1.0J (Japanese)
 2016.06.09 versiom 1.1 with cpu running time data
 2016.12.26 initial process bug fix (all off output at start)
 2017.8.3 make dead time to change open-close for Kindai house
*/

// このプログラムは、無償提供する代わりに、動作結果に関する
// 一切の保証は致しません。自己責任でお使いください。また、
// 改造も自由ですが、公刊等をされる場合は、「近畿大学　星　
// 岳彦が開発した○○〇プログラムを用いた」と引用して下さい
// ますと、今後の研究開発の励みになりますので、どうぞよろ
// しくお願いします。　　　　　2019.3.4 Takehiko Hoshi
// 
// このシールドはArduino MEGA 2560 + Ethernet Shiedld2の
// 組み合わせで動作させることを前提にしています。
// プログラムを書き込むときには、Arduino MEGA 2560だけ
// の状態で書き込んでください。
// Ethernet Sheild 2基板のシールのmacアドレスに変更して
// ください。このプログラムをmacで検索すると付近にあり
// ます。macアドレスとは。90:A2:DA:0F:xx:xx のような8個
// の16進数です。これをばらして0xの後に転記してください。

// 出力の割り付け状態
// Dout0 保温カーテン　開信号
// Dout1 保温カーテン  閉信号
// Dout2 遮光カーテン　開信号
// Dout3 遮光カーテン　閉信号
 
#include <SPI.h>
#include <Ethernet2.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <EEPROM.h>
// UECS用のライブラリUARDECSのインストール
// https://uecs.org/arduino/uardecs.html が必要です。
#include <Uardecs.h>

//
//IPアドレスをリセットするピンの割付
//
//Pin ID. This pin is pull-upped automatically.
const byte U_InitPin = 57 ; // for Arduino MEGA 2560 
//const byte U_InitPin = 17 ; // for Arduino Ethernet
//This shields is assigned as A3 for reset pin.
//When U_InitPin status equals this value,IP address is set "192.168.1.7".
//(This is added Ver3 over)
const byte U_InitPin_Sense= HIGH ;  

//
//　ノードの基礎情報の設定
//
const char U_name[] PROGMEM =   "LU-CURCON-1J";
const char U_vender[] PROGMEM = "Hoshi-lab.";
const char U_uecsid[] PROGMEM = "011009001001";
const char U_footnote[] PROGMEM = "UARDECS7.5 Product by <A HREF=\"http://hoshi-lab.info/\">Kindai, hoshi-lab.</A>";
char U_nodename[20] = "LU-CURCON-1J";
UECSOriginalAttribute U_orgAttribute;

//
// 制御シールドの設定とライブラリ
//
// 入出力ポートの定義
const int port[4] = { 7, 8, 5, 6 } ;
// ウオッチドッグタイマーのリセットポート
const int wdt = 9 ; // 0 = clear timer
// ディジタル出力値の一時記憶装置のトリガポート
const int ratch = 3 ; //  0->1 ratch
// ディジタル入力値のイネーブルポート
const int in_out = 2 ; // in = 1 out =0

// 制御に関する基本的変数
int RelayOut[4] ; // リレー出力変数
int i ; // ループ用変数
signed long hc_openp, sc_openp ; // カーテン開度　単位0.001%　+0.1は開中 +0.9は閉中

// ウオッチドッグタイマーのクリア(2秒以内に1回の実行が必要)
void clearWdt() {
  digitalWrite(wdt, LOW) ;
  delayMicroseconds(100) ;
  digitalWrite(wdt, HIGH) ;
}

// ディジタル値を出力する　d[0]-d[3]の値 0はOFF、1はON
void dout(int *d) {
  digitalWrite(in_out, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], OUTPUT) ;
    digitalWrite(port[i], d[i]) ;
  }
  delayMicroseconds(1) ;
  digitalWrite(ratch, HIGH) ;
  delayMicroseconds(1) ; 
  digitalWrite(ratch, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], INPUT) ;
  }
  digitalWrite(in_out, HIGH) ;
}

// ディジタル出力を全てOFFにする
void clearDout() {
  int d[4] = { 
    LOW, LOW, LOW, LOW
  } ;
  dout(d) ;
  for(i = 0; i < 4; i++) {
    RelayOut[i] = 0 ;
  }
}

//　ディジタル入力値の読み込み　d[0]-d[3]に値を代入
void din(int *d) {
  digitalWrite(in_out, HIGH) ;
  delayMicroseconds(1) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], INPUT) ;
    delayMicroseconds(1) ;
    d[i] = digitalRead(port[i]) ;
  }
}

//////////////////////////////////
// html page setting
//////////////////////////////////

// Define Strings
const char S_Nhc[] PROGMEM =  "保温カーテン";
const char S_Nsc[] PROGMEM =  "遮光カーテン";
const char S_Nop[] PROGMEM =  "動作状況";
const char S_Nmc[] PROGMEM =  "強制動作";
const char S_Nots[] PROGMEM = "室外気温設定";
const char S_Nits[] PROGMEM = "室内気温設定";
const char S_Nhw[] PROGMEM =  "不感帯気温幅";
const char S_Nosr[] PROGMEM = "屋外日射設定";
const char S_Nrt[] PROGMEM =  "日射確認時間";
const char S_Ncd[] PROGMEM =  "夜間強制閉";
const char S_Ndct[] PROGMEM = "暗黒確認時間" ;
const char S_Ntr[] PROGMEM =  "開閉所要時間";
const char S_Nrst[] PROGMEM =  "設定値初期化";
const char S_sec[] PROGMEM = "起動後経過秒"; // WDT動作確認用
//Select Strings
const char S_auto[] PROGMEM = "自動";
const char S_remoteC[] PROGMEM = "CPU遠隔制御";
const char S_remoteS[] PROGMEM = "手動スイッチ";
const char S_web[] PROGMEM = "下記強制動作";
const char S_close[] PROGMEM = "強制閉";
const char S_stop[] PROGMEM = "強制停止";
const char S_open[] PROGMEM = "強制開";
const char S_no[] PROGMEM = "いいえ";
const char S_yes[] PROGMEM = "はい";
const char *sMD[4]={ S_auto, S_remoteC, S_remoteS, S_web };
const char *sOC[3]={ S_close, S_stop, S_open };
const char *sMM[4]={ S_auto, S_close, S_stop, S_open };
const char *sYN[2]={ S_no, S_yes };
//Unit Strings
const char Ucd[] PROGMEM =  "℃";
const char Usrf[] PROGMEM = "kW m-2";
const char Uts[] PROGMEM =  "秒";
//Detail Strings
const char Dop[] PROGMEM =   "-値は閉 +値は開";
const char Dtp1[] PROGMEM =  "外気&GT;設定 かつ";
const char Dtp2[] PROGMEM =  "内気&GT;設定 で開";
const char Dsr[] PROGMEM =   "日射&GT;設定 で閉";
const char Dct[] PROGMEM =   "保温遮光共通" ;
const char Drst[] PROGMEM =  "リセット";
// Dummy Strings
const char EMPTY[] PROGMEM= "";
const char** DUMMY = NULL;

//Define variables for Web interface for heat Insulation Curtain
signed long Wh_mode ; //Th.Ins. Curtain Operation Mode 0:auto 1:remote 2:web
signed long Wh_ope ; // operation mode -:close 0:stop +:open
signed long Wh_ope_p ; // previous operation for pause 1 sec.
signed long Wh_manu ; // manual mode 0:auto 1:close 2:stop 3:open
signed long Wh_spo ; // Outside Airtemperature Setpoint
signed long Wh_spi ; // Inside Airtemperature Setpoint
signed long Wh_hw ; // Hysterisis Width Setpoint
signed long Wh_cdk ; // Close in Darkness
signed long Wh_tr ; // Time Required O/C
//Define variables for Web interface for Shading Curtain
signed long Ws_mode ; //Th.Ins. Curtain Operation Mode 0:auto 1:remote 2:web
signed long Ws_ope ; // operation mode -:close 0:stop +:open
signed long Ws_ope_p ; // previous operation for pause 1 sec.
signed long Ws_manu ; // manual mode 0:auto 1:close 2:stop 3:open
signed long Ws_osr ; // Solar Radiation Flux Setpoint for Close
signed long Ws_rts ; // Retention Time Setpoint
signed long Ws_cdk ; // Close in Darkness
signed long Ws_tr ; // Time Required O/C
//reset
signed long W_rst ; //  Setpoint clear?
//previous mode 前に行った操作を覚えておくため
signed long Wh_manu_p, Ws_manu_p ;
//遮光カーテン基準日射量カウンタ　+基準値を超えた秒数、-基準を下回った秒数
signed long Ws_rts_t ;
//暗黒閉関係の設定値
signed long Whs_dark_t ; // 暗黒になった場合のカウンタ
signed long Whs_dtu ; // Retention Time to Judge in Dark
//カーテン自動制御判定結果 -1:閉めるべき 0:何もしない +1:開けるべき
signed char sc_auto, hc_auto ;
signed long W_sec; // WDT動作確認用

const int U_HtmlLine = 18; //Total number of HTML table rows.
struct UECSUserHtml U_html[U_HtmlLine]={
//{名前,   入出力形式,     単位,    詳細説明, 選択肢文字列,選択肢数, 値,        最小値,最大値, 小数桁数}
  {S_Nhc,  UECSSHOWSTRING, EMPTY,   EMPTY,    sMD,          4,       &(Wh_mode), 0,     3,            0},
  {S_Nop,  UECSSHOWDATA,   Uts,     Dop,      DUMMY,        0,       &(Wh_ope),  0,     0,            0},
  {S_Nmc,  UECSSELECTDATA, EMPTY,   EMPTY,    sMM,          4,       &(Wh_manu), 0,     1,            0},
  {S_Nots, UECSINPUTDATA,  Ucd,     Dtp1,     DUMMY,        0,       &(Wh_spo),  -100,  400,          1},
  {S_Nits, UECSINPUTDATA,  Ucd,     Dtp2,     DUMMY,        0,       &(Wh_spi),  -100,  400,          1},
  {S_Nhw,  UECSINPUTDATA,  Ucd,     EMPTY,    DUMMY,        0,       &(Wh_hw),   0,     100,          1},
  {S_Ncd,  UECSSELECTDATA, EMPTY,   EMPTY,    sYN,          2,       &(Wh_cdk),  0,     1,            0},
  {S_Ndct, UECSINPUTDATA,  Uts,     Dct,      DUMMY,        0,       &(Whs_dtu), 0,     9999,         0},
  {S_Ntr,  UECSINPUTDATA,  Uts,     EMPTY,    DUMMY,        0,       &(Wh_tr),   1,     9999,         0},
  //
  {S_Nsc,  UECSSHOWSTRING, EMPTY,   EMPTY,    sMD,          4,       &(Ws_mode), 0,     3,            0},
  {S_Nop,  UECSSHOWDATA,   Uts,     Dop,      DUMMY,        0,       &(Ws_ope),  0,     0,            0},
  {S_Nmc,  UECSSELECTDATA, EMPTY,   EMPTY,    sMM,          4,       &(Ws_manu), 0,     1,            0},
  {S_Nosr, UECSINPUTDATA,  Usrf,    Dsr,      DUMMY,        0,       &(Ws_osr),  0,     2000,         3},
  {S_Nrt,  UECSINPUTDATA,  Uts,     EMPTY,    DUMMY,        0,       &(Ws_rts),  0,     9999,         0},
  {S_Ncd,  UECSSELECTDATA, EMPTY,   EMPTY,    sYN,          2,       &(Ws_cdk),  0,     1,            0},
  {S_Ntr,  UECSINPUTDATA,  Uts,     EMPTY,    DUMMY,        0,       &(Ws_tr),   1,     9999,         0},
  //
  {S_Nrst, UECSSELECTDATA, EMPTY,   Drst,     sYN,          2,       &(W_rst),   0,     1,            0},
  {S_sec,  UECSSHOWDATA,   Uts,     EMPTY,    DUMMY,        0,       &(W_sec),   0,     0,            0}   // WDT動作確認用
};

//////////////////////////////////
// UserCCM setting
//////////////////////////////////

//define CCMID for identify
//CCMID_dummy must put on last
enum { 
 CCMID_h_cnd,     CCMID_h_opr,    CCMID_h_rcA,      CCMID_h_rcM,
 CCMID_s_cnd,     CCMID_s_opr,    CCMID_s_rcA,      CCMID_s_rcM,
 CCMID_InAirTemp, CCMID_WAirTemp, CCMID_WRadiation, CCMID_Esec,  // WDT動作確認用
 CCMID_dummy
};
const int U_MAX_CCM = CCMID_dummy;
UECSCCM U_ccmList[U_MAX_CCM];

const char ccmNameCnd[] PROGMEM = "NodeCondition";
const char ccmTypeCndh[] PROGMEM = "h_cnd.cCO";
const char ccmTypeCnds[] PROGMEM = "s_cnd.cCO";
const char ccmUnitCnd[] PROGMEM = "";

const char ccmNameOpr[] PROGMEM = "OpenPercentage";
const char ccmTypeOprh[] PROGMEM = "h_opr.cCO";
const char ccmTypeOprs[] PROGMEM = "s_opr.cCO";
const char ccmNamercA[] PROGMEM = "RemoteControl";
const char ccmTypercAh[] PROGMEM = "h_rcA.cCO";
const char ccmTypercAs[] PROGMEM = "s_rcA.cCO";
const char ccmNamercM[] PROGMEM = "ManuralControl";
const char ccmTypercMh[] PROGMEM = "h_rcM.cCO";
const char ccmTypercMs[] PROGMEM = "s_rcM.cCO";
const char ccmUnitP[] PROGMEM = "%";

const char ccmNameITemp[] PROGMEM = "InsideTemperature";
const char ccmTypeITemp[] PROGMEM = "InAirTemp.mIC";
const char ccmNameWTemp[] PROGMEM = "OutsideTemperature";
const char ccmTypeWTemp[] PROGMEM = "WAirTemp.mOC";
const char ccmUnitTemp[] PROGMEM = "C";

const char ccmNameWRad[] PROGMEM = "SolarRadiation";
const char ccmTypeWRad[] PROGMEM = "WRadiation.mOC";
const char ccmUnitRad[] PROGMEM = "kW m-2";

// WDT動作確認用
const char ccmNameEsec[] PROGMEM = "Elapsed time";
const char ccmTypeEsec[] PROGMEM = "esec.cCO";
const char ccmUnitsec[] PROGMEM = "s";


//------------------------------------------------------
// EPROM関係の制御設定値等をクリアする。
// W_rst(リセット)の変数値がおかしい場合もリセット
//------------------------------------------------------
void initializeSetpoints() {
  if(W_rst!=0) {
    Wh_mode = 0 ; //Th.Ins. Curtain Operation Mode 0:auto 1:remote 2:web
    Wh_ope = Wh_ope_p = 0 ; // operation mode -:close 0:stop +:open
    Wh_manu = Wh_manu_p = 0 ; // manual mode 0:auto 1:close 2:stop 3:open
    Wh_spo = 20 ; // Outside Airtemperature Setpoint
    Wh_spi = 160 ; // Inside Airtemperature Setpoint
    Wh_hw = 40 ; // Hysterisis Width Setpoint
    Wh_cdk = 1 ; // Close in Darkness
    Whs_dtu = 900 ; // Retention Time to Judge in Dark
    Wh_tr = 300 ; // Time Required O/C
    Ws_mode = 0 ; //Th.Ins. Curtain Operation Mode 0:auto 1:remote 2:web
    Ws_ope = Ws_ope_p = 0 ; // operation mode -:close 0:stop +:open
    Ws_manu = Ws_manu_p = 0 ; // manual mode 0:auto 1:close 2:stop 3:open
    Ws_osr = 200 ; // Solar Radiation Flux Setpoint for Close
    Ws_rts = 900 ; // Retention Time Setpoint
    Ws_cdk = 1 ; // Close in Darkness
    Ws_tr = 300 ; // Time Required O/C  
    W_rst = 0 ; // Setpoint clear?
  }

}

//------------------------------------------------------
// カーテン開閉のための定時作業(1秒に1回呼び出す必要あり)
// Wh_ope と Ws_ope に基づいて行う
//------------------------------------------------------
#define HCUR_OPEN 0
#define HCUR_CLOSE 1
#define SCUR_OPEN 2
#define SCUR_CLOSE 3

void curtainRoutine() {
  signed long hc_pu, sc_pu ; 
  //開閉パーセント計算
  hc_pu = 100000 / Wh_tr ;
  sc_pu = 100000 / Ws_tr ;

  if((Wh_ope>0)&&(Wh_ope * Wh_ope_p >= 0)) {
    //保温カーテン開 1回前との積が負になったら1秒休む
    RelayOut[HCUR_CLOSE] = 0 ;
    RelayOut[HCUR_OPEN] = 1 ;
    Wh_ope_p = Wh_ope ;
    hc_openp = hc_openp + hc_pu ;
    if(hc_openp > 99000) hc_openp = 100000 ;
    Wh_ope-- ;
  }
  else if((Wh_ope<0)&&(Wh_ope * Wh_ope_p >= 0)) {
    //保温カーテン閉 1回前との積が負になったら1秒休む
    RelayOut[HCUR_OPEN] = 0 ;
    RelayOut[HCUR_CLOSE] = 1 ;
    Wh_ope_p = Wh_ope ;
    hc_openp = hc_openp - hc_pu ;
    if(hc_openp < 1000) hc_openp = 0 ;
    Wh_ope++ ;
  }
  else {
    //保温カーテン停止
    RelayOut[HCUR_OPEN] = 0 ;
    RelayOut[HCUR_CLOSE] = 0 ;
    Wh_ope_p = Wh_ope ;
  }
  if((Ws_ope>0)&&(Ws_ope * Ws_ope_p >= 0)) {
    //遮光カーテン開 1回前との積が負になったら1秒休む
    RelayOut[SCUR_CLOSE] = 0 ;
    RelayOut[SCUR_OPEN] = 1 ;
    Ws_ope_p = Ws_ope ;
    sc_openp = sc_openp + sc_pu ;
    if(sc_openp > 99000) sc_openp = 100000 ;
    Ws_ope-- ;
  }
  else if((Ws_ope<0)&&(Ws_ope * Ws_ope_p >= 0)) {
    //遮光カーテン閉 1回前との積が負になったら1秒休む
    RelayOut[SCUR_OPEN] = 0 ;
    RelayOut[SCUR_CLOSE] = 1 ;
    Ws_ope_p = Ws_ope ;
    sc_openp = sc_openp - sc_pu ;
    if(sc_openp < 1000) sc_openp = 0 ;
    Ws_ope++ ;
  }
  else {
    //遮光カーテン停止
    RelayOut[SCUR_OPEN] = 0 ;
    RelayOut[SCUR_CLOSE] = 0 ;
    Ws_ope_p = Ws_ope ;
  }
  // リレー出力
  dout(RelayOut) ;
}

//------------------------------------------------------
// 手動操作が指定されたときに所定の動作を実施する
// 強制開閉のときは、保持時間だけW*_opeに値セット
//------------------------------------------------------
void manualOperation() {
  //保温カーテン
  switch(Wh_manu) {
    case 1: //強制閉
      Wh_mode = 3 ; //Webモード
      if(Wh_manu_p != Wh_manu) { // もし今回初めての強制閉の指示なら
        Wh_ope = -Wh_tr ; //保持時間だけ閉信号セット
        Wh_manu_p = Wh_manu ;
      }
      break ;
    case 2: //強制停止
      Wh_mode = 3 ; //Webモード
      Wh_ope = 0 ; //停止信号セット
      Wh_manu_p = Wh_manu ;
      break ;
    case 3:
       Wh_mode = 3 ; //Webモード
      if(Wh_manu_p != Wh_manu) { // もし今回初めての強制開の指示なら
        Wh_ope = Wh_tr ; //保持時間だけ開信号セット
        Wh_manu_p = Wh_manu ;
      }
      break ;
    default:
      Wh_mode = 0 ; //AUTOモード
      Wh_ope = 0 ; //停止信号セット
      Wh_manu_p = Wh_manu = 0 ; //念のためマニュアル値を正しい値に矯正
  }
  //遮光カーテン
  switch(Ws_manu) {
    case 1: //強制閉
      Ws_mode = 3 ; //Webモード
      if(Ws_manu_p != Ws_manu) { // もし今回初めての強制閉の指示なら
        Ws_ope = -Ws_tr ; //保持時間だけ閉信号セット
        Ws_manu_p = Ws_manu ;
      }
      break ;
    case 2: //強制停止
      Ws_mode = 3 ; //Webモード
      Ws_ope = 0 ; //停止信号セット
      Ws_manu_p = Ws_manu ;
      break ;
    case 3:
       Ws_mode = 3 ; //Webモード
      if(Ws_manu_p != Ws_manu) { // もし今回初めての強制開の指示なら
        Ws_ope = Ws_tr ; //保持時間だけ開信号セット
        Ws_manu_p = Ws_manu ;
      }
      break ;
    default:
      Ws_mode = 0 ; //AUTOモード
      Ws_ope = 0 ; //停止信号セット
      Ws_manu_p = Ws_manu = 0 ; //念のためマニュアル値を正しい値に矯正
  }
  
}

//------------------------------------------------------
// ネットからのリモート操作 h_rcM.cCOとs_rcM.cCOのCCM
// 有効のとき、-1: リモート操作解除、
//             0% 閉信号を3秒セット
//             100% 開信号を3秒セット
//             1-99% +/-5%以上離れていれば、それを満たす方向に3秒セット
//------------------------------------------------------
void rcMOperation() {
  // 保温カーテン
  // モードがWebの強制動作より下の場合だけ受付
  if(Wh_mode < 3) {
    // リモートCCMが無効でリモート操作モードだったら解除
    if(!(U_ccmList[CCMID_h_rcM].validity)) {
      if(Wh_mode == 2) {
        Wh_mode = 0 ; //AUTOモード
        Wh_ope = 0 ; //停止信号セット
      }
    }
    // リモートCCMが有効だったら処理
    else {
      // rcMが-1でリモート操作モードだったら解除
      if(U_ccmList[CCMID_h_rcM].value == -1) {
        if(Wh_mode == 2) {
          Wh_mode = 0 ; //AUTOモード
          Wh_ope = 0 ; //停止信号セット
        }
      }
      else {
        // rcMが0-100%の意味ある値であれば
        if((U_ccmList[CCMID_h_rcM].value >= 0)&&(U_ccmList[CCMID_h_rcM].value <= 100)) {
          Wh_mode = 2 ; // リモート操作モードに
          // それぞれの値で処理
          switch(U_ccmList[CCMID_h_rcM].value) {
            case 0:
              Wh_ope = -3 ; // 3秒閉
              break ;
            case 100:
              Wh_ope = 3 ; // 3秒開
              break ;
            default:
              // リモート操作の値と開度との差を計算
              // △-5%以下ならば開く
              if(((hc_openp / 1000) - U_ccmList[CCMID_h_rcM].value) < -5) Wh_ope = 3 ; // 3秒開
              // △5%以上ならば閉じる
              else if(((hc_openp / 1000) - U_ccmList[CCMID_h_rcM].value) > 5) Wh_ope = -3 ; // 3秒閉
              else Wh_ope = 0 ; // それ以外なら停止
          }
        }
      }
    }
  }
  // 遮光カーテン
  // モードがWebの強制動作より下の場合だけ受付
  if(Ws_mode < 3) {
    // リモートCCMが無効でリモート操作モードだったら解除
    if(!(U_ccmList[CCMID_s_rcM].validity)) {
      if(Ws_mode == 2) {
        Ws_mode = 0 ; //AUTOモード
        Ws_ope = 0 ; //停止信号セット
      }
    }
    // リモートCCMが有効だったら処理
    else {
      // rcMが-1でリモート操作モードだったら解除
      if(U_ccmList[CCMID_s_rcM].value == -1) {
        if(Ws_mode == 2) {
          Ws_mode = 0 ; //AUTOモード
          Ws_ope = 0 ; //停止信号セット
        }
      }
      else {
        // rcMが0-100%の意味ある値であれば
        if((U_ccmList[CCMID_s_rcM].value >= 0)&&(U_ccmList[CCMID_s_rcM].value <= 100)) {
          Ws_mode = 2 ; // リモート操作モードに
          // それぞれの値で処理
          switch(U_ccmList[CCMID_s_rcM].value) {
            case 0:
              Ws_ope = -3 ; // 3秒閉
              break ;
            case 100:
              Ws_ope = 3 ; // 3秒開
              break ;
            default:
              // リモート操作の値と開度との差を計算
              // △-5%以下ならば開く
              if(((sc_openp / 1000) - U_ccmList[CCMID_s_rcM].value) < -5) Ws_ope = 3 ; // 3秒開
              // △5%以上ならば閉じる
              else if(((sc_openp / 1000) - U_ccmList[CCMID_s_rcM].value) > 5) Ws_ope = -3 ; // 3秒閉
              else Ws_ope = 0 ; // それ以外なら停止 
          }
        }
      }
    }
  }
}

//------------------------------------------------------
// ネットからのリモート制御 h_rcA.cCOとs_rcA.cCOのCCM
// 有効のとき、-1: リモート操作解除、
//             0% 閉信号を60秒セット
//             100% 開信号を60秒セット
//             1-99% +/-5%以上離れていれば、それを満たす方向に60秒セット
//------------------------------------------------------
void rcAOperation() {
  // 保温カーテン
  // モードがリモート制御より下の場合だけ受付
  if(Wh_mode < 2) {
    // リモートCCMが無効でリモート制御モードだったら解除
    if(!(U_ccmList[CCMID_h_rcA].validity)) {
      if(Wh_mode == 1) {
        Wh_mode = 0 ; //AUTOモード
        Wh_ope = 0 ; //停止信号セット
      }
    }
    // リモートCCMが有効だったら処理
    else {
      // rcMが-1でリモート制御モードだったら解除
      if(U_ccmList[CCMID_h_rcA].value == -1) {
        if(Wh_mode == 1) {
          Wh_mode = 0 ; //AUTOモード
          Wh_ope = 0 ; //停止信号セット
        }
      }
      else {
        // rcMが0-100%の意味ある値であれば
        if((U_ccmList[CCMID_h_rcA].value >= 0)&&(U_ccmList[CCMID_h_rcA].value <= 100)) {
          Wh_mode = 1 ; // リモート制御モードに
          // それぞれの値で処理
          switch(U_ccmList[CCMID_h_rcA].value) {
            case 0:
              Wh_ope = -60 ; // 3秒閉
              break ;
            case 100:
              Wh_ope = 60 ; // 3秒開
              break ;
            default:
              // リモート制御の値と開度との差を計算
              // △-5%以下ならば開く
              if(((hc_openp / 1000) - U_ccmList[CCMID_h_rcA].value) < -5) Wh_ope = 60 ; // 60秒開
              // △5%以上ならば閉じる
              else if(((hc_openp / 1000) - U_ccmList[CCMID_h_rcA].value) > 5) Wh_ope = -60 ; // 60秒閉
              else Wh_ope = 0 ;  // それ以外なら停止
          }
        }
      }
    }
  }
  // 遮光カーテン
  // モードがリモート制御より下の場合だけ受付
  if(Ws_mode < 2) {
    // リモートCCMが無効でリモート制御モードだったら解除
    if(!(U_ccmList[CCMID_s_rcA].validity)) {
      if(Ws_mode == 1) {
        Ws_mode = 0 ; //AUTOモード
        Ws_ope = 0 ; //停止信号セット
      }
    }
    // リモートCCMが有効だったら処理
    else {
      // rcMが-1でリモート制御モードだったら解除
      if(U_ccmList[CCMID_s_rcA].value == -1) {
        if(Ws_mode == 1) {
          Ws_mode = 0 ; //AUTOモード
          Ws_ope = 0 ; //停止信号セット
        }
      }
      else {
        // rcMが0-100%の意味ある値であれば
        if((U_ccmList[CCMID_s_rcA].value >= 0)&&(U_ccmList[CCMID_s_rcA].value <= 100)) {
          Ws_mode = 1 ; // リモート制御モードに
          // それぞれの値で処理
          switch(U_ccmList[CCMID_s_rcA].value) {
            case 0:
              Ws_ope = -60 ; // 3秒閉
              break ;
            case 100:
              Ws_ope = 60 ; // 3秒開
              break ;
            default:
              // リモート制御の値と開度との差を計算
              // △-5%以下ならば開く
              if(((sc_openp / 1000) - U_ccmList[CCMID_s_rcA].value) < -5) Ws_ope = 60 ; // 60秒開
              // △5%以上ならば閉じる
              else if(((sc_openp / 1000) - U_ccmList[CCMID_s_rcA].value) > 5) Ws_ope = -60 ; // 60秒閉
              else Ws_ope = 0 ;  // それ以外なら停止
          }
        }
      }
    }
  }
}

//------------------------------------------------------
// 自動制御 autoモードのときの判定主プログラム
// 保温カーテン用hcCond()と遮光カーテン用scCond()の
// サブルーチンを呼び出し
//------------------------------------------------------
void autoControl(){
  // 暗黒条件でカーテンを閉めるかどうかの判断
  // 日射データが無効ならばカウンタクリア
  if(!(U_ccmList[CCMID_WRadiation].validity)) Whs_dark_t = 0 ;
  //そうでなければカウント カウント絶対値が上限値DarkTupを超えた時にカウントストップ
  else {
    // 0.001 kW m-2 以下ならばカウントアップ 暗黒継続時正値、明継続時負値
    if(U_ccmList[CCMID_WRadiation].value <= 1) {
      if(Whs_dark_t > 0) {
        Whs_dark_t++ ;
        if(Whs_dark_t > Whs_dtu) {
          Whs_dark_t = Whs_dtu ;
        }
      }
      else Whs_dark_t = 1 ;
    }
    else {
      if(Whs_dark_t < 0) {
        Whs_dark_t--;
        if(-Whs_dark_t > Whs_dtu) {
          Whs_dark_t = -Whs_dtu ;
        }
      }
      else Whs_dark_t = -1 ;
    }
  }
  // 保温カーテン制御
  // 自動制御モードになっていて、
  if(Wh_mode == 0) {
    // 暗黒で閉める条件ならば閉める、それ以外は保温条件でチェック
    if(Wh_cdk == 1) {
      if(Whs_dark_t >= Whs_dtu) hc_auto = -1 ;
      else if(-Whs_dark_t >= Whs_dtu) hcCond() ;
    }
    else hcCond() ;
    // カーテンを閉めるべきで開度が1%(10)以上の時には、閉信号を2秒出す
    // すぐ後の実行ルーチン(curtainRoutine())で1秒分減らされるので2秒にする
    if((hc_auto == -1)&&(U_ccmList[CCMID_h_opr].value >= 10)) Wh_ope = -2 ;
    // カーテンを開けるべきで開度が100%(1000)未満の時には、開信号を2秒出す
    // 全閉で閉めているときには0.9%、全開で空けているときには100.1%になるので
    if((hc_auto == 1)&&(U_ccmList[CCMID_h_opr].value < 1000)) Wh_ope = 2 ;    
  }
  // 遮光カーテン制御
  // 自動制御モードになっていて
  if(Ws_mode == 0) {
    // 暗黒で閉める条件ならば閉める、それ以外は遮光条件でチェック
    if(Ws_cdk == 1) {
      if(Whs_dark_t >= Whs_dtu) sc_auto = -1 ;
      else if(-Whs_dark_t >= Whs_dtu) scCond() ;
    }
    else scCond() ;
    // カーテンを閉めるべきで開度が1%(10)以上の時には、閉信号を2秒出す
    // すぐ後の実行ルーチン(curtainRoutine())で1秒分減らされるので2秒にする
    if((sc_auto == -1)&&(U_ccmList[CCMID_s_opr].value >= 10)) Ws_ope = -2 ;
    // カーテンを開けるべきで開度が100%(1000)未満の時には、開信号を2秒出す
    // 全閉で閉めているときには0.9%、全開で空けているときには100.1%になるので
    if((sc_auto == 1)&&(U_ccmList[CCMID_s_opr].value < 1000)) Ws_ope = 2 ;
  }    
}  
  
//------------------------------------------------------
// 保温カーテン用自動制御
// 室内気温が設定値を下回り、かつ、室外気温が設定値を下回った
// ときに閉める、それ以外は開けるが、ヒステリシスがある
//------------------------------------------------------
void hcCond(){
  // 保温カーテン
  // 開ける条件 - 室内と室外の気温が両方とも設定を上回る
  // 室内気温が有効で
  if(U_ccmList[CCMID_InAirTemp].validity) {
    // 室内気温が(設定気温+ヒステリシス/2)を超えて
    if(U_ccmList[CCMID_InAirTemp].value > (Wh_spi + Wh_hw / 2)) {
      // 室外気温が有効で
      if(U_ccmList[CCMID_WAirTemp].validity) {
        // 室外気温が(設定気温+ヒステリシス/2)を超えた場合だけ保温カーテンを開ける
        if(U_ccmList[CCMID_WAirTemp].value > (Wh_spo + Wh_hw / 2)) hc_auto = 1 ;
       }
       // 室外気温が無効のときは、室内気温の開条件を満足しただけで開ける
       else hc_auto = 1 ;
     }
  }
  else {
    // 室内気温が無効の場合は室外気温だけで判断
    // 室外気温が有効で
    if(U_ccmList[CCMID_WAirTemp].validity) {
      // 室外気温が(設定気温+ヒステリシス/2)を超えたら保温カーテンを開ける
      if(U_ccmList[CCMID_WAirTemp].value > (Wh_spo + Wh_hw / 2)) hc_auto = 1 ;
    }
  }
  // 閉める条件 - 室内または室外の少なくとも一つが設定を下回る
  // 室内気温が有効で室内気温が(設定気温-ヒステリシス/2)を下回ったらカーテンを閉める
  if((U_ccmList[CCMID_InAirTemp].validity)&&
    (U_ccmList[CCMID_InAirTemp].value < (Wh_spi - Wh_hw / 2))) hc_auto = -1 ;
  // 室外気温が有効で室外気温が(設定気温-ヒステリシス/2)を下回ったらカーテンを閉める
  if((U_ccmList[CCMID_WAirTemp].validity)&&
    (U_ccmList[CCMID_WAirTemp].value < (Wh_spo - Wh_hw / 2))) hc_auto = -1 ;
}

//------------------------------------------------------
// 遮光カーテン用自動制御
// 日射量が一定時間設定値を超過した場合カーテン閉める
//------------------------------------------------------
void scCond(){
  // 遮光カーテン
  // 日射データが無効ならばカウンタクリア
  if(!(U_ccmList[CCMID_WRadiation].validity)) Ws_rts_t = 0 ;
  // 有効ならば、遮光日射設定値と比較しカウントアップ
  else {
    // 日射が設定値を超えたとき
    if(U_ccmList[CCMID_WRadiation].value > Ws_osr) {
      if(Ws_rts_t > 0) {
        Ws_rts_t++ ; // カウントアップ
        // カウント値が規定を超えたらカーテンを閉めるべき
        if(Ws_rts_t > Ws_rts) {
          Ws_rts_t = Ws_rts ;
          sc_auto = -1 ;
        }
      }
      // 初めて超えたときのセット
      else Ws_rts_t = 1 ;
    }
    // 日射が設定値以下の時
    else {
      if(Ws_rts_t <= 0) {
        Ws_rts_t-- ; // カウントダウン
        // カウント減算値が規定を超えたらカーテンを開けるべき
        if(-Ws_rts_t > Ws_rts) {
          Ws_rts_t = -Ws_rts ;
          sc_auto = 1 ;
        }
      }
      // 初めて設定値以下のときのセット
      else Ws_rts_t = -1 ;
    }
  }
}
  
//------------------------------------------------------
// CCMに値をセットする。
// 毎秒チェック
//------------------------------------------------------
void setCCM(){
  // cndに制御モード値をセット
  U_ccmList[CCMID_h_cnd].value = Wh_mode ;
  U_ccmList[CCMID_s_cnd].value = Ws_mode ;
  // oprに開度をセット
  U_ccmList[CCMID_h_opr].value = hc_openp / 1000 * 10 ;
  if(Wh_ope > 0) U_ccmList[CCMID_h_opr].value += 1 ;
  if(Wh_ope < 0) U_ccmList[CCMID_h_opr].value += 9 ;  
  U_ccmList[CCMID_s_opr].value = sc_openp / 1000 * 10 ;
  if(Ws_ope > 0) U_ccmList[CCMID_s_opr].value += 1 ;
  if(Ws_ope < 0) U_ccmList[CCMID_s_opr].value += 9 ; 
  W_sec =(signed long)(millis() / 1000) ;  // WDT動作確認用
  U_ccmList[CCMID_Esec].value = W_sec ;  // WDT動作確認用
}
  
//------------------------------------------------------
//UARDECS初期化用関数
//主にCCMの作成とMACアドレスの設定を行う
//------------------------------------------------------
void UserInit(){

 //Set ccm list
 UECSsetCCM(true,  CCMID_h_cnd,      ccmNameCnd,   ccmTypeCndh,  ccmUnitCnd,  30, 0, A_1S_0 );
 UECSsetCCM(true,  CCMID_h_opr,      ccmNameOpr,   ccmTypeOprh,  ccmUnitP,    30, 1, A_10S_0);
 UECSsetCCM(false, CCMID_h_rcA,      ccmNamercA,   ccmTypercAh,  ccmUnitP,    30, 0, A_1M_0 );
 UECSsetCCM(false, CCMID_h_rcM,      ccmNamercM,   ccmTypercMh,  ccmUnitP,    30, 0, A_1S_0 );
 UECSsetCCM(true,  CCMID_s_cnd,      ccmNameCnd,   ccmTypeCnds,  ccmUnitCnd,  30, 0, A_1S_0 );
 UECSsetCCM(true,  CCMID_s_opr,      ccmNameOpr,   ccmTypeOprs,  ccmUnitP,    30, 1, A_10S_0);
 UECSsetCCM(false, CCMID_s_rcA,      ccmNamercA,   ccmTypercAs,  ccmUnitP,    30, 0, A_1M_0 );
 UECSsetCCM(false, CCMID_s_rcM,      ccmNamercM,   ccmTypercMs,  ccmUnitP,    30, 0, A_1S_0 );
 UECSsetCCM(false, CCMID_InAirTemp,  ccmNameITemp, ccmTypeITemp, ccmUnitTemp, 30, 1, A_10S_0);
 UECSsetCCM(false, CCMID_WAirTemp,   ccmNameWTemp, ccmTypeWTemp, ccmUnitTemp, 30, 1, A_10S_0);
 UECSsetCCM(false, CCMID_WRadiation, ccmNameWRad,  ccmTypeWRad,  ccmUnitRad,  30, 3, A_10S_0);
 UECSsetCCM(true,  CCMID_Esec,       ccmNameEsec,  ccmTypeEsec,  ccmUnitsec,  30, 0, A_1S_0);  // WDT動作確認用

 //Set proper MAC address
 U_orgAttribute.mac[0] = 0x90;
 U_orgAttribute.mac[1] = 0xA2;
 U_orgAttribute.mac[2] = 0xDA;
 U_orgAttribute.mac[3] = 0x0F;
 U_orgAttribute.mac[4] = 0xEB;
 U_orgAttribute.mac[5] = 0x3D;
}

//---------------------------------------------------------
//Webページから入力が行われ各種値を設定し不揮発性メモリに値を保存後、
//ページの再描画を行う前に以下の関数が呼び出される。
//---------------------------------------------------------
void OnWebFormRecieved(){

  initializeSetpoints();
  manualOperation(); // Web操作処理

}

void UserEverySecond(){

  clearWdt() ;
  rcMOperation() ; // リモート操作処理
  rcAOperation() ; // リモート制御処理
  autoControl() ; // 自律自動制御処理
  curtainRoutine(); // 1単位のカーテン操作
  setCCM() ; // CCMの値計算

}

//---------------------------------------------------------
//１分に１回呼び出される関数
//---------------------------------------------------------
void UserEveryMinute(){
}

//---------------------------------------------------------
//メインループ
//システムのタイマカウント，httpサーバーの処理，
//UDP16520番ポートと16529番ポートの通信文をチェックした後，呼び出さされる関数。
//呼び出される頻度が高いため，重い処理を記述しないこと。
//---------------------------------------------------------
void UserEveryLoop(){
}

//---------------------------------------------------------
//setup()実行後に呼び出されるメインループ
//この関数内ではUECSloop()関数を呼び出さなくてはならない。
//UserEveryLoop()に似ているがネットワーク関係の処理を行う前に呼び出される。
//必要に応じて処理を記述してもかまわない。
//呼び出される頻度が高いため,重い処理を記述しないこと。
//---------------------------------------------------------
void loop(){
  UECSloop();
}


//---------------------------------------------------------
//起動直後に１回呼び出される関数。
//様々な初期化処理を記述できる。
//この関数内ではUECSsetup()関数を呼び出さなくてはならない。
//必要に応じて処理を記述してもかまわない。
//---------------------------------------------------------
void setup(){

  //
  //制御シールドの制御信号線の初期化
  //
  pinMode(in_out, OUTPUT) ;
  digitalWrite(in_out, LOW) ;
  for(i = 0; i < 4; i++) {
    pinMode(port[i], OUTPUT) ;
    digitalWrite(port[i], LOW) ;
  }
  pinMode(ratch, OUTPUT) ;
  digitalWrite(ratch, HIGH) ;
  delayMicroseconds(1) ; 
  digitalWrite(ratch, LOW) ;
  pinMode(wdt, OUTPUT) ;
  
  digitalWrite(wdt, HIGH) ; // ウオッチドッグクリア信号を初期値にする
  clearWdt() ;
  clearDout() ; // 出力をリセット

  UECSsetup();
  
  initializeSetpoints();
  // カーテン開度は全開に
  hc_openp = sc_openp = 100000 ;
  // 日射量カウンタリセット
  Ws_rts_t = Whs_dark_t = 0 ;

}
