/*
 *  Arduino SensorShield for LOW-COST UECS by T.Hoshi
 *  LU-ENVMEAS-1 created on 2016.3.14
 *  LU-ENVMEAS-2 changed on 2017.8.8
 *  LU-ENVMEAS-3 for S-300G CO2 sensor on 2017.11.15
 *  LU-ENVMEAS-4 for Solar Rad Sensor change 2019.2.22
 *   OPL20A25101 and S-300G (0-2000 ppm 0.5-4.5V)
 *  LU-ENVMEAS-5 for SHT-31 usable on 2019.8.27
 *  LU-ENVMEAS-6 for S-300G org. switchable on 2020.2.25
 *  LU-ENVMEAS-7 for new solar panel SY-M0.5W on 2020.7.6
 *  LU-ENVMEAS71 Reset cycle of WDT are shorten to avoid 
 *   awaking the watch-dog in the setup() process on 2022.11.18
 *  LU-ENVMEAS-8 By discontunue solar cells, two other cells, 
 *   LR0GC02 and SY-M1.15, are aveilable on 2023.11.28
 *  LU-ENVMEAS-9 TYPE=time A_1M -> A_10S to avoid CCM loss for  
 *   recording software 2024.2.20
-----------------------------------------------------------------------------
LU-ENVMEAS-9 software are distributed under the 2 clause BSD license.

Copyright 2024, Takehiko Hoshi All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice, 
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.

-----------------------------------------------------------------------------
LU-ENVMEAS-9ソフトウェアは、2条項BSDライセンスの下に配布されます。

Copyright 2024, Takehiko Hoshi All rights reserved.

  ソースコード形式かバイナリ形式か、変更するかしないかを問わず、以下の条件を
満たす場合に限り、再頒布および使用が許可されます。

1.ソースコードを再頒布する場合、上記の著作権表示、本条件一覧、および下記免責
  条項を含めること。
2.バイナリ形式で再頒布する場合、頒布物に付属のドキュメント等の資料に、上記の
  著作権表示、本条件一覧、および下記免責条項を含めること。

  本ソフトウェアは、著作権者およびコントリビューターによって「現状のまま」提供
されており、明示黙示を問わず、商業的な使用可能性、および特定の目的に対する適合
性に関する暗黙の保証も含め、またそれに限定されない、いかなる保証もありません。
著作権者もコントリビューターも、事由のいかんを問わず、 損害発生の原因いかんを
問わず、かつ責任の根拠が契約であるか厳格責任であるか（過失その他の）不法行為で
あるかを問わず、仮にそのような損害が発生する可能性を知らされていたとしても、
本ソフトウェアの使用によって発生した （代替品または代用サービスの調達、使用の
喪失、データの喪失、利益の喪失、業務の中断も含め、またそれに限定されない）直接
損害、間接損害、偶発的な損害、特別損害、懲罰的損害、または結果損害について、
一切責任を負わないものとします。
-----------------------------------------------------------------------------
*/
// このプログラムは、無償提供する代わりに、動作結果に関する
// 一切の保証は致しません。自己責任でお使いください。また、
// 改造も自由ですが、公刊等をされる場合は、「近畿大学　星　
// 岳彦が開発した○○〇プログラムを用いた」と引用して下さい
// ますと、今後の研究開発の励みになりますので、どうぞよろ
// しくお願いします。　　　　　2024.2.20 Takehiko Hoshi
// 
// このシールドはArduino MEGA 2560 + Ethernet Shiedld2の
// 組み合わせで動作させることを前提にしています。
// プログラムを書き込むときには、Arduino MEGA 2560だけ
// の状態で書き込んでください。
// Ethernet Sheild 2基板のシールのmacアドレスに変更して
// ください。このプログラムをmacで検索すると付近にあり
// ます。macアドレスとは。90:A2:DA:0F:xx:xx のような8個
// の16進数です。これをばらして0xの後に転記してください。
// もし、CO2濃度の計測値が低めに出てしまうとき、
// フルスケール2000ppmのS-300GUを使ってるか確認を。
// そうではなく、S-300Gをフルスケール10000ppmで使っているなら
// 以下のFSCO2の#defineの行を変更して下さい。

#include <Wire.h>
#include <SPI.h>
#include <SD.h>
#include <Ethernet2.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <EEPROM.h>
// UECS用のライブラリUARDECSのインストール
// https://uecs.org/arduino/uardecs.html が必要です。
#include <Uardecs_mega.h>

// Start up title
const char sut[17] = "240220 ENVMEAS-9" ;

// Arduino Ethernet Shield2のMacアドレス
//必ずシールドに貼ってあるアドレスに書き換えて使用して下さい
const byte ma[6] = { 0xA8, 0x61, 0x0A, 0xAE, 0x06, 0x9F } ;

// 以下は計測信号線の定義
//センサ・SD信号線定義
//analog
const uint8_t TH0Pin  =  0 ; //サーミスタ0 地温
const uint8_t TH1Pin  =  1 ; //サーミスタ1 室外気温
const uint8_t ppfPin  =  2 ; //太陽電池 光関係
const uint8_t CO2Pin  =  3 ; //CO2センサ
//Digital
const int chipSelect = 4 ; //SDカード
const uint8_t PBPin  =  6 ; //押しボタンスイッチ
const uint8_t WDTPin = 9 ;
// I2Cデバイスアドレスの定義
#define LCD_ADRS  0x3E //液晶表示器
#define RTC_ADRS  0x32 //リアルタイムクロック
#define SHT31_ADRS 0x45 // SHT31

// ■CO2センサの型番による補正 以下のいずれかをアクティブに
// S-300GUの場合はF.S. 2000 ppm
#define FSCO2 2000
// S-300GUでないS-300Gのフルスケール10000 ppm場合は次の行を使う
// #define FSCO2 10000

// ■日射束計測用　太陽電池の選択 for ver.9
// 型番定義
enum {
  OPL20A25101, // シャント抵抗6.8Ωのまま　温度補正係数　-0.45 %/℃
  SY_M05W, // SUNYOO solar  SY-M0.5W シャント抵抗6.8Ωのまま　温度補正係数　-0.50 %/℃
  LR0GC02, // SHARP LR0GC02 シャント抵抗6.8Ωのまま　温度補正係数　-0.32 %/℃
  // https://akizukidenshi.com/catalog/g/gM-06564/
  SY_M115W // SUNYOO solar SY-M1.15W シャント抵抗3.3Ωに変更　温度補正係数　-0.50 %/℃
  // https://akizukidenshi.com/catalog/g/gM-08918/
} ;
// 使用する太陽電池セルを上記からSolarCellに設定
const char SolarCell = SY_M05W ;
//                     ↑ココ
  
// UARDECS用IP初期化　ショートピン(D3)を抜くとセーフモード(192.168.1.7)
const byte U_InitPin = 3; // digital D3 pin for Safe mode
const byte U_InitPin_Sense=HIGH;

// UARDECSのUECSノード情報設定
const char U_name[] PROGMEM =   "LU-ENVMEAS-8";
const char U_vender[] PROGMEM = "Hoshi-lab.";
// このUECS-IDは近畿大学のものです。販売等される際にはUECS研究会から各自取得してください。
const char U_uecsid[] PROGMEM = "011009000001";
const char U_footnote[] PROGMEM = "Product by <A HREF=\"http://hoshi-lab.info/\">Kindai, hoshi-lab.</A>";
char U_nodename[20] = "LU-ENVMEAS-8";
UECSOriginalAttribute U_orgAttribute;

//計測ノード固有の変数定義
char fln[] = "LOGDATA.csv" ;
File fp ;
boolean sdcond ;
byte seconds,minutes,pminutes,hours,days,months,years;
char dates[9] = "00/00/00", times[9] = "00:00:00"  ;
char LCDbuf[20] ;
float mv ;
float m_it, m_ot, m_ist, m_irh , m_iah, m_idp, m_isdf, m_osrf, m_oppf, m_ico2 ;
int dispNo, sw, count10 ;
//SHT31データ用変数
uint8_t sht_rd[6] ;

//UECS WEB テーブル定義
//表示トークン定義
const char EMPTY[] PROGMEM= "";
const char** DUMMY = NULL;
const char t_date[] PROGMEM =  "日付";
const char u_date[] PROGMEM =  "yymmdd";
const char t_time[] PROGMEM =  "時刻";
const char u_time[] PROGMEM =  "hhmm";
const char t_RTCset[] PROGMEM =  "日時設定";
const char t_DTsend[] PROGMEM =  "日時CCM送出";
const char t_DO[] PROGMEM =  "する";
const char t_NON[] PROGMEM =  "しない";
const char *sDO[4]={ t_NON, t_DO };
const char t_co2adj[] PROGMEM =  "CO2補正値";
const char u_co2[] PROGMEM =  "ppm";
const char c_co2adj[] PROGMEM =  "センサ値+補正値";
//値格納用変数
signed long v_date ; //日付
signed long v_time ; //時刻
signed long v_RTCset ; //日時設定
signed long v_DTsend ; //日時CCM送出
signed long v_co2adj ; //CO2オフセット補正値

//Webページの表の定義
const int U_HtmlLine = 5; //Total number of HTML table rows.
struct UECSUserHtml U_html[U_HtmlLine]={
//{名前,   入出力形式,     単位,    詳細説明, 選択肢文字列,選択肢数,値,        最小値,最大値, 小数桁数}
  {t_date,  UECSINPUTDATA,  u_date, EMPTY,    DUMMY,       0,       &(v_date),  0,     999999,       0},
  {t_time,  UECSINPUTDATA,  u_time, EMPTY,    DUMMY,       0,       &(v_time),  0,     9999,         0},
  {t_RTCset,UECSSELECTDATA, EMPTY,  EMPTY,    sDO,         2,       &(v_RTCset),1,     1,            0},
  {t_DTsend,UECSSELECTDATA, EMPTY,  EMPTY,    sDO,         2,       &(v_DTsend),0,     1,            0},
  {t_co2adj,UECSINPUTDATA,  u_co2,  c_co2adj, DUMMY,       0,       &(v_co2adj),-5000, 5000,         0}
};

//UECS CCM 定義
enum {
CCMID_Time,
CCMID_Date,
CCMID_InAirTemp,
CCMID_InAirHumid,
CCMID_InDewPoint,
CCMID_InAbsHumid,
CCMID_InSatDiffHumid,
CCMID_InAirCO2,
CCMID_WRadiation,
CCMID_WPPF,
CCMID_InSoilTemp,
CCMID_WAirTemp,
CCMID_cnd,
CCMID_dummy,
};

const int U_MAX_CCM = CCMID_dummy;
UECSCCM U_ccmList[U_MAX_CCM];

const char ccmNameTime[] PROGMEM = "時刻";
const char ccmNameDate[] PROGMEM = "日付";
const char ccmNameInAirTemp[] PROGMEM = "室内気温";
const char ccmNameInAirHumid[] PROGMEM = "室内相対湿度";
const char ccmNameInDewPoint[] PROGMEM = "室内露点温度";
const char ccmNameInAbsHumid[] PROGMEM = "室内絶対湿度";
const char ccmNameInSatDiffHumid[] PROGMEM = "室内飽差";
const char ccmNameInAirCO2[] PROGMEM = "室内CO2濃度";
const char ccmNameWRadiation[] PROGMEM = "室外日射束";
const char ccmNameWPPF[] PROGMEM = "室外光量子束";
const char ccmNameInSoilTemp[] PROGMEM = "室内培地温度";
const char ccmNameWAirTemp[] PROGMEM = "室外気温";
const char ccmNamecnd[] PROGMEM = "機器状態";

const char ccmTypeTime[] PROGMEM = "Time";
const char ccmTypeDate[] PROGMEM = "Date";
const char ccmTypeInAirTemp[] PROGMEM = "InAirTemp.mIC";
const char ccmTypeInAirHumid[] PROGMEM = "InRelativeHumid.mIC";
const char ccmTypeInDewPoint[] PROGMEM = "InDewPoint.mIC";
const char ccmTypeInAbsHumid[] PROGMEM = "InAbsHumid.mIC";
const char ccmTypeInSatDiffHumid[] PROGMEM = "InAirSatDef.mIC";
const char ccmTypeInAirCO2[] PROGMEM = "InAirCO2.mIC";
const char ccmTypeWRadiation[] PROGMEM = "WRadiation.mOC";
const char ccmTypeWPPF[] PROGMEM = "WPPF.mOC";
const char ccmTypeInSoilTemp[] PROGMEM = "InSoilTemp.mIC";
const char ccmTypeWAirTemp[] PROGMEM = "WAirTemp.mOC";
const char ccmTypecnd[] PROGMEM = "cnd.mIC.mOC";

const char ccmUnitTime[] PROGMEM = "HHMM";
const char ccmUnitDate[] PROGMEM = "YYMMDD";
const char ccmUnitTemp[] PROGMEM = "C";
const char ccmUnitHumid[] PROGMEM = "%";
const char ccmUnitAbsHumid[] PROGMEM = "g/kg'";
const char ccmUnitAirCO2[] PROGMEM = "ppm";
const char ccmUnitRadiation[] PROGMEM = "kW/m2";
const char ccmUnitPPF[] PROGMEM = "umol/m2/s";
const char ccmUnitcnd[] PROGMEM = "";

void UserInit(){
 //Set proper MAC address
 U_orgAttribute.mac[0] = ma[0];
 U_orgAttribute.mac[1] = ma[1];
 U_orgAttribute.mac[2] = ma[2];
 U_orgAttribute.mac[3] = ma[3];
 U_orgAttribute.mac[4] = ma[4];
 U_orgAttribute.mac[5] = ma[5];

 //Set ccm list
 // sender, ID, Description, Type name, Unit, Priority, Decimal Digit, CCM Level
 UECSsetCCM(true, CCMID_Time, ccmNameTime, ccmTypeTime, ccmUnitTime, 29, 0, A_10S_0);
 UECSsetCCM(true, CCMID_Date, ccmNameDate, ccmTypeDate, ccmUnitDate, 29, 0, A_1M_0);
 UECSsetCCM(true, CCMID_InAirTemp, ccmNameInAirTemp, ccmTypeInAirTemp, ccmUnitTemp, 29, 1, A_10S_0);
 UECSsetCCM(true, CCMID_InAirHumid, ccmNameInAirHumid, ccmTypeInAirHumid, ccmUnitHumid, 29, 0, A_10S_0);
 UECSsetCCM(true, CCMID_InDewPoint, ccmNameInDewPoint, ccmTypeInDewPoint, ccmUnitTemp, 29, 1, A_10S_0);
 UECSsetCCM(true, CCMID_InAbsHumid, ccmNameInAbsHumid, ccmTypeInAbsHumid, ccmUnitAbsHumid, 29, 1, A_10S_0);
 UECSsetCCM(true, CCMID_InSatDiffHumid, ccmNameInSatDiffHumid, ccmTypeInSatDiffHumid, ccmUnitAbsHumid, 29, 1, A_10S_0);
 UECSsetCCM(true, CCMID_InAirCO2, ccmNameInAirCO2, ccmTypeInAirCO2, ccmUnitAirCO2, 29, 0, A_10S_0);
 UECSsetCCM(true, CCMID_WRadiation, ccmNameWRadiation, ccmTypeWRadiation, ccmUnitRadiation, 29, 3, A_10S_0);
 UECSsetCCM(true, CCMID_WPPF, ccmNameWPPF, ccmTypeWPPF, ccmUnitPPF, 29, 1, A_10S_0);
 UECSsetCCM(true, CCMID_InSoilTemp, ccmNameInSoilTemp, ccmTypeInSoilTemp, ccmUnitTemp, 29, 1, A_10S_0);
 UECSsetCCM(true, CCMID_WAirTemp, ccmNameWAirTemp, ccmTypeWAirTemp, ccmUnitTemp, 29, 1, A_10S_0);
 UECSsetCCM(true, CCMID_cnd, ccmNamecnd, ccmTypecnd, ccmUnitcnd, 29, 0, A_1S_0);
}

void setup(){
  
// Serial.begin(9600); // for debug
// Serial.print("*1\n");

  // ウオッチドッグタイマーをリセット
  pinMode(WDTPin, OUTPUT) ; 
  clearWdt() ;

  // For UARDECS
  UECSsetup();
//Serial.print("*2\n");

  // 押しボタンのI/O設定
  pinMode(PBPin, INPUT);
  
  // ウオッチドッグタイマーのクリア
  clearWdt() ;

  //初期化が必要なCCM valures
  U_ccmList[CCMID_cnd].value=0;
  
  // 表示切替、PB検出変数などの初期化
  dispNo = 0 ;
  sw = 0 ;
  count10 = 0 ;
  
  // I2C初期化
  Wire.begin();
  
  // RTC初期化
  Wire.beginTransmission(RTC_ADRS);
  Wire.write(0xE0);
  Wire.write(0x20);  //24hourMode
  Wire.write(0x00);  // PON Clear
  Wire.endTransmission();
  delay(1);
  minutes = 0 ; // initialize previous logging time

  // ウオッチドッグタイマーのクリア
  clearWdt() ;

/*
 // RTCを起動時に一定日時にセットする場合
  Wire.beginTransmission(RTC_ADRS);
  Wire.write(0x00);
  Wire.write(0x00);  //sec
  Wire.write(0x00);  //min
  Wire.write(0x12);  //hour
  Wire.write(0x00);  //week 0:日曜　１:月曜　・・・　6:土曜
  Wire.write(0x01);  //day
  Wire.write(0x01);  //month
  Wire.write(0x16);  //year
  Wire.endTransmission();
  delay(1);
*/

  //SDファイル作成
  sdcond = SD.begin(chipSelect) ; // if error sdcond == 0
//Serial.print("sdcond = " );  //for debug
//Serial.println(sdcond); //for debug
  if(sdcond) SDwriteHead() ;

// ウオッチドッグタイマーのクリア
  clearWdt() ;
  
//Serial.print("*3\n");

  //LCD初期化
  LCD_command(0x38);
  LCD_command(0x39);
  LCD_command(0x14);
  LCD_command(0x73);
  LCD_command(0x52);  // 5V
//LCD_command(0x56);  // 3V
  LCD_command(0x6C);
  LCD_command(0x38);
  LCD_command(0x01);
  LCD_command(0x0C);

  // SHT31を初期化
  SHT31_command(0x30A2) ; // Soft reset
  SHT31_command(0x3041) ; // Soft reset
  SHT31_command(0x3066) ; // Heater off

  //リアルタイムクロックから時刻読み込み
  RTCread() ;

  // ウオッチドッグタイマーのクリア
  clearWdt() ;

  //日時の液晶表示 起動時にすぐ出すため
  LCDdateDisp() ;
  // スタートアップ　クレジット表示
  LCD_command(0x80|0x40);
  int ss;
  for(ss = 0; ss < 16; ss++) LCD_write(sut[ss]);

  // ウオッチドッグタイマーのクリア
  clearWdt() ;
//Serial.print("*4\n");
  
}

// ウオッチドッグタイマーのクリア(2秒以内に1回の実行が必要)
void clearWdt() {
  digitalWrite(WDTPin, HIGH) ;
  delay(5);
  digitalWrite(WDTPin, LOW) ;
}

//RTCから日時読み込み
void RTCread() {
  pminutes = minutes ;
  Wire.requestFrom(RTC_ADRS,8);
  Wire.read();
  seconds = Wire.read();
  minutes = Wire.read();
  hours = Wire.read();
  Wire.read();
  days = Wire.read();
  months = Wire.read();
  years = Wire.read();
  
  //CCMに日時をセット
  U_ccmList[CCMID_Time].value = ((long)(hours >> 4) * 10 + (long)(hours & 0xF)) * 10000 + ((long)(minutes >> 4) * 10 + (long)(minutes & 0xF)) * 100 ;
  U_ccmList[CCMID_Date].value = ((long)(years >> 4) * 10 + (long)(years & 0xF)) * 10000 + ((long)(months >> 4) * 10 + (long)(months & 0xF)) * 100 + (long)(days >> 4) * 10 + (long)(days & 0xF) ;

  //Web設定変数に値セット
  v_date = U_ccmList[CCMID_Date].value ;
  v_time = U_ccmList[CCMID_Time].value / 100 ;
}

//RTC日時書き込み
void RTCset() {
  byte bbuf ;
  Wire.beginTransmission(RTC_ADRS);
  Wire.write(0x00);
  Wire.write(0x00);  //sec
  bbuf = (byte)(v_time % 10) + (byte)(v_time / 10 % 10 * 16) ; //1と10の位
//Serial.print("bbuf1 = " );  //for debug
//Serial.println(bbuf); //for debug
  Wire.write(bbuf);  //min
  bbuf = (byte)(v_time / 100 % 10) + (byte)(v_time / 1000 % 10 * 16) ; //100と1000の位
  Wire.write(bbuf);  //hour
  Wire.write(0x00);  //week 0:日曜　１:月曜　・・・　6:土曜
  bbuf = (byte)(v_date % 10) + (byte)(v_date / 10 % 10 * 16) ; //1と10の位
  Wire.write(bbuf);  //day
  bbuf = (byte)(v_date / 100 % 10) + (byte)(v_date / 1000 % 10 * 16) ; //100と1000の位
  Wire.write(bbuf);  //month
  bbuf = (byte)(v_date / 10000 % 10) + (byte)(v_date / 100000 % 10 * 16) ; //10000と100000の位
  Wire.write(bbuf);  //year
  Wire.endTransmission();
  delay(1);
}

//LCDにデータを送るサブルーチン
void LCD_write(byte t_data) {
  Wire.beginTransmission(LCD_ADRS);
  Wire.write(0x40);
  Wire.write(t_data);
  Wire.endTransmission();
  delay(1);
}
 
//LCDに命令を実行させるサブルーチン
void LCD_command(byte t_command)
{
  Wire.beginTransmission(LCD_ADRS);
  Wire.write(0x00);
  Wire.write(t_command);
  Wire.endTransmission();
  delay(10);
}
 
//液晶1行目に日時を表示
void LCDdateDisp() {
  LCD_command(0x80|0x00);
   
  LCD_write('0'+(years>>4));
  LCD_write('0'+(years&0xF));
  LCD_write('/');
  LCD_write('0'+(months>>4));
  LCD_write('0'+(months&0xF));
  LCD_write('/');
  LCD_write('0'+(days>>4));
  LCD_write('0'+(days&0xF));
  LCD_write(' ');
  LCD_write(' ');
  LCD_write('0'+(hours>>4));
  LCD_write('0'+(hours&0xF));
  LCD_write(':');
  LCD_write('0'+(minutes>>4));
  LCD_write('0'+(minutes&0xF));

}

//液晶2行目に計測値を表示 dn 表示するデータ項目番号
void LCDvalueDisp(int dn) {
  int i ;
  //LCD下段バッファクリア
  for(i = 0; i <16; i++) {
    LCDbuf[i] = ' ' ;    
  }
  LCDbuf[16] = 0 ;

  //モードに応じた表示内容
  switch(dn) {
    case 0: 
            LCDbuf[0] ='I' ;
            LCDbuf[1] ='n' ;
            LCDbuf[2] =' ' ;
            LCDbuf[3] ='A' ;
            LCDbuf[4] ='i' ;
            LCDbuf[5] ='r' ;
            LCDbuf[6] =' ' ;
            LCDbuf[7] ='T' ;
            LCDbuf[8] ='=' ;
            dtostrf(m_it, 5, 1, &LCDbuf[9]) ;
            LCDbuf[14] = 0xdf ;
            LCDbuf[15] = 'C' ;
            break ;
    case 1: 
            LCDbuf[0] ='S' ;
            LCDbuf[1] ='o' ;
            LCDbuf[2] ='i' ;
            LCDbuf[3] ='l' ;
            LCDbuf[4] =' ' ;
            LCDbuf[5] ='T' ;
            LCDbuf[6] ='.' ;
            LCDbuf[7] ='=' ;
            dtostrf(m_ist, 5, 1, &LCDbuf[8]) ;
            LCDbuf[13] = 0xdf ;
            LCDbuf[14] = 'C' ;
            break ;
    case 2: 
            LCDbuf[0] ='O' ;
            LCDbuf[1] ='.' ;
            LCDbuf[2] =' ' ;
            LCDbuf[3] ='A' ;
            LCDbuf[4] ='i' ;
            LCDbuf[5] ='r' ;
            LCDbuf[6] =' ' ;
            LCDbuf[7] ='T' ;
            LCDbuf[8] ='=' ;
            dtostrf(m_ot, 5, 1, &LCDbuf[9]) ;
            LCDbuf[14] = 0xdf ;
            LCDbuf[15] = 'C' ;
            break ;
    case 3:
            LCDbuf[0] ='R' ;
            LCDbuf[1] ='a' ;
            LCDbuf[2] ='d' ;
            LCDbuf[3] ='=' ;
            dtostrf(m_osrf, 5, 3, &LCDbuf[5]) ;
            LCDbuf[11] = 'k' ;
            LCDbuf[12] = 'W' ;
            LCDbuf[13] = '/' ;
            LCDbuf[14] = 'm' ;
            LCDbuf[15] = '2' ;
            break ;
    case 4:
            LCDbuf[0] ='P' ;
            LCDbuf[1] ='P' ;
            LCDbuf[2] ='F' ;
            LCDbuf[3] ='=' ;
            dtostrf(m_oppf, 6, 1, &LCDbuf[4]) ;
            LCDbuf[11] = 'u' ;
            LCDbuf[12] = 'm' ;
            LCDbuf[13] = 'o' ;
            LCDbuf[14] = 'l' ;
            break ;
    case 5:
            LCDbuf[0] ='C' ;
            LCDbuf[1] ='O' ;
            LCDbuf[2] ='2' ;
            LCDbuf[3] ='=' ;
            dtostrf(m_ico2, 4, 0, &LCDbuf[4]) ;
            LCDbuf[9] = 'p' ;
            LCDbuf[10] = 'p' ;
            LCDbuf[11] = 'm' ;
            break ;
    case 6:
            LCDbuf[0] ='R' ;
            LCDbuf[1] ='.' ;
            LCDbuf[2] ='H' ;
            LCDbuf[3] ='.' ;
            LCDbuf[4] ='=' ;
            dtostrf(m_irh, 5, 1, &LCDbuf[4]) ;
            LCDbuf[10] = '%' ;
            break ;
    case 7:
            LCDbuf[0] ='D' ;
            LCDbuf[1] ='e' ;
            LCDbuf[2] ='w' ;
            LCDbuf[3] ='P' ;
            LCDbuf[4] ='.' ;
            LCDbuf[5] ='=' ;
            dtostrf(m_idp, 5, 1, &LCDbuf[6]) ;
            LCDbuf[11] = 0xdf ;
            LCDbuf[12] = 'C' ;
            break ;
    case 8:
            LCDbuf[0] ='A' ;
            LCDbuf[1] ='.' ;
            LCDbuf[2] ='H' ;
            LCDbuf[3] ='.' ;
            LCDbuf[4] ='=' ;
            dtostrf(m_iah, 5, 1, &LCDbuf[5]) ;
            LCDbuf[11] = 'g' ;
            LCDbuf[12] = '/' ;
            LCDbuf[13] = 'k' ;
            LCDbuf[14] = 'g' ;
            LCDbuf[15] = 0xf4 ;
            break ;
    case 9:
            LCDbuf[0] ='S' ;
            LCDbuf[1] ='.' ;
            LCDbuf[2] ='D' ;
            LCDbuf[3] ='.' ;
            LCDbuf[4] ='=' ;
            dtostrf(m_isdf, 5, 1, &LCDbuf[5]) ;
            LCDbuf[11] = 'g' ;
            LCDbuf[12] = '/' ;
            LCDbuf[13] = 'k' ;
            LCDbuf[14] = 'g' ;
            LCDbuf[15] = 0xf4 ;
            break ;
     }

  //LCD下段表示
  LCD_command(0x80|0x40);
  for(i=0; i<16; i++) {
    LCD_write(LCDbuf[i]) ;
  }
}

//SHT31に命令を実行させる
void SHT31_command(uint16_t cmd)
{
  Wire.beginTransmission(SHT31_ADRS);
  Wire.write(cmd>>8); 
  Wire.write(cmd&0xFF);
  Wire.endTransmission(); 
  delay(300) ; 
}

//SHT31tから計測データを読み込む
void SHT31_read(void)
{
  int i,j ;
  Wire.requestFrom(SHT31_ADRS,6);
  for(i = 0 ; i < 100 ; i++) { // Wait for Data
    if(Wire.available() == 6) break ; // data ready
    delay(10) ;
  }
  if(i < 100) { // data read
      for(j = 0; j < 6; j++) sht_rd[j] = Wire.read() ;
  }
  else { // Time out
      sht_rd[0]= 212 ;  sht_rd[1]= 28 ; // 100℃
      sht_rd[3]= 0 ;  sht_rd[4]= 0 ; // 0%
  }
}

//SDメモリにファイルをオープンし、ヘッダを書き込む
void SDwriteHead() {
  fp = SD.open(fln, FILE_WRITE);
  if(fp) { // if successfully open file
    fp.print("Date(yyyy/mm/dd), Time(hh:mm:ss), In Air Temp.(C), In R.H.(%), In D.P.(C),") ;
    fp.print("In Abs.H.(g/kgDA), In S.Diff.(g/kgDA), In Air CO2(ppm), Out Solar R.F.(kW/m2),") ;
    fp.println(" Out PPF(umol), In Soil Temp.(C), Out Air Temp.(C)") ;
    fp.close() ;
  }
}

//日時と共に1回分のデータをSDに書き込む
void SDwriteRecord() {    
    dates[0] = '0'+(years>>4) ;
    dates[1] = '0'+(years&0xF) ;
    dates[3] = '0'+(months>>4) ;
    dates[4] = '0'+(months&0xF) ;
    dates[6] = '0'+(days>>4) ;
    dates[7] = '0'+(days&0xF) ;
    times[0] = '0'+(hours>>4) ;
    times[1] = '0'+(hours&0xF) ;
    times[3] = '0'+(minutes>>4) ;
    times[4] = '0'+(minutes&0xF) ;
    times[6] = '0' ;
    times[7] = '0' ;

    fp = SD.open(fln, FILE_WRITE);
    if(fp) { // if successfully open file
      fp.print(dates) ;
      fp.print(", ") ;
      fp.print(times) ;
      fp.print(", ") ;
      fp.print(m_it) ;
      fp.print(", ") ;
      fp.print(m_irh) ;
      fp.print(", ") ;
      fp.print(m_idp) ;
      fp.print(", ") ;
      fp.print(m_iah) ;
      fp.print(", ") ;
      fp.print(m_isdf) ;
      fp.print(", ") ;
      fp.print(m_ico2) ;
      fp.print(", ") ;
      fp.print(m_osrf) ;
      fp.print(", ") ;
      fp.print(m_oppf) ;
      fp.print(", ") ;
      fp.print(m_ist) ;
      fp.print(", ") ;
      fp.println(m_ot) ;
      fp.close() ;
    }
}

//水蒸気圧を求める
float CalcEP(float t, float rh) {
  // Obtain absolute humidity at air temperature(t) and relative humidity (rh)
  float a1,a2,a3,a4,a5,a6,b1,b2,b3,e0,atc,atemp,ps,es,ep ;
  // constants
  a1 = 10.79574 ;
  a2 = -5.028 ;
  a3 = 1.50475e-4 ;
  a4 = -8.2969 ;
  a5 = 0.42873e-3 ;
  a6 = 4.76955 ;
  b1 = -9.09685 ;
  b2 = -3.56654 ;
  b3 = 0.87682 ;
  e0 = 4.581 ;
  atc = 273.15 ;
  atemp = atc + t ;
  // icing case division by air temperature
  if(t >= 0) {
    ps = a1 * (1.0 - atc / atemp) + a2 * log10(atemp / atc)
      + a3 * (1.0 - pow(10.0, a4 * (atemp / atc - 1.0)))
        + a5 * (pow(10.0, a6 * (1.0 - atc / atemp)) - 1.0)
          + log10(e0) ;
  }
  else {
    ps = b1 * (atc / atemp - 1.0) + b2 * log10(atc / atemp)
      + b3 * (1.0 - atc / atemp) + log10(e0) ;
  }
  // get saturated vapour pressure mmHg
  es = pow(10.0, ps) ;
  // get vapour pressior part by relative humidity
  ep = es * rh / 100.0 ;
  return ep ;  
}

//絶対湿度計算
float CalcAH(float t, float rh) {
  // Obtain absolute humidity at air temperature(t) and relative humidity (rh)
  float ep, ah ;
  ep = CalcEP(t, rh) ;
  // get saturated absoulte humidity  g kg'-1
  ah = (0.622 * ep / (760.0 - ep) * 1000.0) ;
  return ah ;    
}

//露点温度計算
float CalcDP(float t, float rh) {
  // Obtain dewpoint temperature at air temperature(t) and relative humidity (rh)
  float ep, dp ;
  ep = CalcEP(t, rh) ;
  // convert mmHg to hPa
  ep = ep * 1.333224 ;
  // get dewpoint temperature humidity  g kg'-1
  dp = 237.3 * log10(6.1078 / ep) / (log10(ep / 6.1078) - 7.5) ;
  return dp ;    
}
  
//環境計測処理
void envSens() {
  // サーミスター0計測
  mv = (float)analogRead(TH0Pin) *  5 / 1023 ;
  if((mv > 4.8)||(mv < 0.9)) { // temp. zone  -19,21 C to +64.37 C
    m_ist = 9999 ;
    U_ccmList[CCMID_InSoilTemp].flagStimeRfirst = false ; // データ送出停止
    U_ccmList[CCMID_cnd].value=-10;
  }
  else {
    m_ist = (1.3679 * mv - 29.229) * mv + 89.57 ;
  }
  U_ccmList[CCMID_InSoilTemp].value =(long)(m_ist * 10) ;

  // サーミスター1計測
  mv = (float)analogRead(TH1Pin) *  5 / 1023 ;
  if((mv > 4.8)||(mv < 0.9)) { // temp. zone  -19,21 C to +64.37 C
    m_ot = 9999 ;
    U_ccmList[CCMID_WAirTemp].flagStimeRfirst = false ; // データ送出停止
    U_ccmList[CCMID_cnd].value=-100;
  }
  else {
    m_ot = (1.3679 * mv - 29.229) * mv + 89.57 ;
  }
  U_ccmList[CCMID_WAirTemp].value =(long)(m_ot * 10) ;

  // 太陽電池の日射束計測
  mv = (float)analogRead(ppfPin) *  5000 / 1023 ;
  if(mv < 0) mv = 0.0 ;
  // 外気温が測定できた場合のパネル温度を25℃に補正した出力電圧
  // 太陽電池型番ごとの補正値の設定 for ver.8
  float tempCoef ;
  switch(SolarCell) {
    case OPL20A25101:
      tempCoef = 0.0045 ;
      break ;
    case LR0GC02:
      tempCoef = 0.0032 ;
      break ;
    default:
      tempCoef = 0.0050 ;
  }
  // 外気温が正常範囲内で計測できていれば、電圧の温度補正
  if((m_ot >= -10.0)&&(m_ot <= 50.0)) mv = mv - (25 - m_ot) * tempCoef ;
  // 日射フラックス密度換算式 mV -> kW/m2 (6.8　OHM for shunt R) for ver.8
  switch(SolarCell) {
    case OPL20A25101:
      m_osrf = 5.6800E-04 * mv ;
      break ;
    case LR0GC02:
      m_osrf = 1.72862E-03 * mv ;
      break ;
    case SY_M115W: // シャント抵抗3.3Ωに変更
      m_osrf = 9.50228E-04 * mv ;
      break ;
   default: // SY_M05W including
      m_osrf = 5.18423E-04 * mv ;
  }
  if(m_osrf < 0) m_osrf = 0.0 ;
  U_ccmList[CCMID_WRadiation].value =(long)(m_osrf * 1000) ;
  // ppf convert
  m_oppf =  m_osrf * 1667 ; // PPFDへの換算
  U_ccmList[CCMID_WPPF].value =(long)(m_oppf * 10) ;
  
  // CO2計測 v_co2adjは補正値
  // S-300G 5V CO2 sensor: FS 2000ppm 0.5 - 4.5 V
  m_ico2 = (float)analogRead(CO2Pin) *  5000 / 1024 ; // mV
  // - 500mV OFFSET * CO2F.S. / 4000mV FS + Ajdust value
  m_ico2 = (m_ico2 - 500) * FSCO2 / 4000 + (float)v_co2adj ;
  U_ccmList[CCMID_InAirCO2].value =(long)m_ico2 ;
  
  // 温度・湿度を計測 SHT31
  // SHT31から温湿度データを取得
  SHT31_command(0x2400) ; // 計測命令
  SHT31_read() ; // 値の読み込み
  // 気温換算 CRC検証なし
  m_it = -45.0 + (175.0 * ((float)sht_rd[0] * 256.0 + (float)sht_rd[1]) / 65535.0) ;
  U_ccmList[CCMID_InAirTemp].value =(long)(m_it * 10) ;
  // 相対湿度換算 CRC検証なし
  m_irh = (100.0 * ((float)sht_rd[3] * 256.0 + (float)sht_rd[4]) / 65535.0) ;
  U_ccmList[CCMID_InAirHumid].value =(long)m_irh ;
  // 露点温度計算
  m_idp = CalcDP(m_it, m_irh) ;
  U_ccmList[CCMID_InDewPoint].value =(long)(m_idp * 10) ;
  //絶対湿度計算
  m_iah = CalcAH(m_it, m_irh) ;
  U_ccmList[CCMID_InAbsHumid].value =(long)(m_iah * 10) ;
  //飽差計算  
  m_isdf = CalcAH(m_it, 100.0) - m_iah ;
  U_ccmList[CCMID_InSatDiffHumid].value =(long)(m_isdf * 10) ;
}

//ループ処理
void loop() {
  // For UARDECS
  UECSloop();
}

//Webのフォームで値がセットされたときに実行
void OnWebFormRecieved(){
  //リアルタイムクロックセット
  if(v_RTCset == 1) {
    RTCset() ;
    //セットした日時を表示
    RTCread() ;
    LCDdateDisp() ;
    v_RTCset = 0 ;   
  }
  else {
    //Web設定変数をCCMの値にリセット
    v_date = U_ccmList[CCMID_Date].value ;
    v_time = U_ccmList[CCMID_Time].value / 100 ;
  }

  //DATE&TIME CCM 送出オンオフ
  //v_DTsend に値が入るので、UserEverySecond()内で処理

}

//1秒ごとのループ
void UserEverySecond(){

  clearWdt() ; // reset WDT

  //if内は10秒に1回の実行
  if(++count10 >= 10) {
    //10秒に1回測定
    count10 = 0 ;
    //環境測定
    envSens() ;
    //データ表示
    LCDvalueDisp(dispNo) ;
    
    //date time CCM の送出禁止処理(一時停止で)
    if(v_DTsend == 0) {
      // ↑秒ごとにtrueが上書きされるのでtrue処理を書く必要はない
       U_ccmList[CCMID_Time].flagStimeRfirst = false ;
       U_ccmList[CCMID_Date].flagStimeRfirst = false ;
     }
   }
}

//1分ごとのループ
void UserEveryMinute(){

  //リアルタイムクロックから時刻読み込み
  RTCread() ;
  //日時の液晶表示
  LCDdateDisp() ;
  //SDに書き込み
//Serial.print("sdcond = " );  //for debug
//Serial.println(sdcond); //for debug
  if(sdcond) SDwriteRecord() ;
}

//ユーザー用ループ
void UserEveryLoop(){
  
  //押しボタンスイッチが押されたときの処理
  if(sw==0) { // 押しボタンスイッチが押されたかテスト
    if(digitalRead(PBPin) == 0) {
      sw = 1;  //押しボタンスイッチ検出フラグセット　連続防止
      //表示するデータをカウントアップ
      if(++dispNo >= 10) dispNo = 0 ;
      LCDvalueDisp(dispNo) ; // 新しい値を液晶に表示
    }
  }
  else { // 押しボタンスイッチが放されたかテスト
    if(digitalRead(PBPin) == 1) {
      sw = 0;  //押しボタンスイッチ検出フラグクリア　連続防止
    }
  }
}
